/*
 * Asterisk -- A telephony toolkit for Linux.
 *
 * Implementation of Media Gateway Control Protocol
 * 
 * Copyright (C) 1999, Mark Spencer
 *
 * Mark Spencer <markster@linux-support.net>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */

#include <stdio.h>
#include <pthread.h>
#include <string.h>
#include <asterisk/lock.h>
#include <asterisk/channel.h>
#include <asterisk/channel_pvt.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/options.h>
#include <asterisk/lock.h>
#include <asterisk/sched.h>
#include <asterisk/io.h>
#include <asterisk/rtp.h>
#include <asterisk/acl.h>
#include <asterisk/callerid.h>
#include <asterisk/cli.h>
#include <asterisk/say.h>
#include <asterisk/cdr.h>
#include <asterisk/astdb.h>
#include <asterisk/parking.h>
#include <asterisk/app.h>
#include <asterisk/musiconhold.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/signal.h>
#include <signal.h>
#include <pthread.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#include <asterisk/dsp.h>
#include <ctype.h>

#ifndef IPTOS_MINCOST
#define IPTOS_MINCOST 0x02
#endif

#define MGCPDUMPER
#define DEFAULT_EXPIREY 120
#define MAX_EXPIREY     3600
#define CANREINVITE	1

static char *desc = "Media Gateway Control Protocol (MGCP)";
static char *type = "MGCP";
static char *tdesc = "Media Gateway Control Protocol (MGCP)";
static char *config = "mgcp.conf";

#define DEFAULT_MGCP_PORT	2427/* From RFC 2705 */
#define MGCP_MAX_PACKET	1500		/* Also from RFC 2543, should sub headers tho */

/* MGCP rtp stream modes */
#define MGCP_CX_SENDONLY 0
#define MGCP_CX_RECVONLY 1
#define MGCP_CX_SENDRECV 2
#define MGCP_CX_CONF     3
#define MGCP_CX_CONFERENCE 3
#define MGCP_CX_MUTE     4
#define MGCP_CX_INACTIVE 4

static char *mgcp_cxmodes[] = {
    "sendonly",
    "recvonly",
    "sendrecv",
    "confrnce",
    "inactive"
};

static char context[AST_MAX_EXTENSION] = "default";

static char language[MAX_LANGUAGE] = "";
static char musicclass[MAX_LANGUAGE] = "";
static char callerid[AST_MAX_EXTENSION] = "";

static int inbanddtmf = 0;
static int nat = 0;

/* Not used. Dosn't hurt for us to always send cid  */
/* to the mgcp box. */
/* static int use_callerid = 1;*/
/*static int cur_signalling = -1;*/

/*static unsigned int cur_group = 0;*/
static unsigned int cur_callergroup = 0;
static unsigned int cur_pickupgroup = 0;

/* XXX Is this needed? */
/*     Doesn't look like the dsp stuff for */
/*     inbanddtmf is actually hooked up.   */
/* static int relaxdtmf = 0; */

static int tos = 0;

static int immediate = 0;

static int callwaiting = 0;

/* Not used. Dosn't hurt for us to always send cid  */
/* to the mgcp box. */
/*static int callwaitingcallerid = 0;*/

/*static int hidecallerid = 0;*/

static int callreturn = 0;

static int threewaycalling = 0;

/* This is for flashhook transfers */
static int transfer = 0;

static int cancallforward = 0;

static int canreinvite = CANREINVITE;

/*static int busycount = 3;*/

/*static int callprogress = 0;*/

static char accountcode[20] = "";

static char mailbox[AST_MAX_EXTENSION];

static int amaflags = 0;

static int adsi = 0;


static int usecnt =0;
static ast_mutex_t usecnt_lock = AST_MUTEX_INITIALIZER;
static int oseq;

/* Wait up to 16 seconds for first digit (FXO logic) */
static int firstdigittimeout = 16000;

/* How long to wait for following digits (FXO logic) */
static int gendigittimeout = 8000;

/* How long to wait for an extra digit, if there is an ambiguous match */
static int matchdigittimeout = 3000;

/* Protect the monitoring thread, so only one process can kill or start it, and not
   when it's doing something critical. */
static ast_mutex_t netlock = AST_MUTEX_INITIALIZER;

static ast_mutex_t monlock = AST_MUTEX_INITIALIZER;

/* This is the thread for the monitor which checks for input on the channels
   which are not currently in use.  */
static pthread_t monitor_thread = 0;

static int restart_monitor(void);

/* Just about everybody seems to support ulaw, so make it a nice default */
static int capability = AST_FORMAT_ULAW;
static int nonCodecCapability = AST_RTP_DTMF;

static char ourhost[256];
static struct in_addr __ourip;
static int ourport;

static int mgcpdebug = 0;

static struct sched_context *sched;
static struct io_context *io;
/* The private structures of the  mgcp channels are linked for
   selecting outgoing channels */
   
#define MGCP_MAX_HEADERS		64
#define MGCP_MAX_LINES 		64

struct mgcp_request {
	int len;
	char *verb;
	char *identifier;
	char *endpoint;
	char *version;
	int headers;					/* MGCP Headers */
	char *header[MGCP_MAX_HEADERS];
	int lines;						/* SDP Content */
	char *line[MGCP_MAX_LINES];
	char data[MGCP_MAX_PACKET];
};

static struct mgcp_pkt {
	int retrans;
	struct mgcp_endpoint *owner;
	int packetlen;
	char data[MGCP_MAX_PACKET];
	struct mgcp_pkt *next;
} *packets = NULL;	

/* MGCP message for queuing up */
struct mgcp_message {
	unsigned int seqno;
	int len;
	struct mgcp_message *next;
	unsigned char buf[0];
};

#define MAX_SUBS 2

#define SUB_REAL 0
#define SUB_ALT  1

struct mgcp_subchannel {
	ast_mutex_t lock;
    int id;
    struct ast_channel *owner;
    struct mgcp_endpoint *parent;
    struct ast_rtp *rtp;
	struct sockaddr_in tmpdest;
	char txident[80];
    char cxident[80];
    char callid[80];
    time_t lastouttime;
    int lastout;
    int cxmode;
	int nat;
	int iseq; /* Not used? RTP? */
	int outgoing;
	int alreadygone;
	int messagepending;
	struct mgcp_message *msgs;			/* Message queue */
    struct mgcp_subchannel *next; /* for out circular linked list */
};

#define MGCP_ONHOOK 1
#define MGCP_OFFHOOK 2

#define TYPE_TRUNK		1
#define TYPE_LINE		2

struct mgcp_endpoint {
	ast_mutex_t lock;
	char name[80];
    struct mgcp_subchannel *sub;        /* pointer to our current connection, channel and stuff */
	char accountcode[80];
	char exten[AST_MAX_EXTENSION];		/* Extention where to start */
	char context[AST_MAX_EXTENSION];
	char language[MAX_LANGUAGE];
	char callerid[AST_MAX_EXTENSION];	/* Caller*ID */
	char lastcallerid[AST_MAX_EXTENSION];	/* Last Caller*ID */
	char call_forward[AST_MAX_EXTENSION];	/* Last Caller*ID */
    char mailbox[AST_MAX_EXTENSION];
    char musicclass[MAX_LANGUAGE];
	char curtone[80];					/* Current tone */
    unsigned int callgroup;
    unsigned int pickupgroup;
	int callwaiting;
    int transfer;
    int threewaycalling;
	int cancallforward;
	int canreinvite;
	int callreturn;
	int dnd; /* How does this affect callwait?  Do we just deny a mgcp_request if we're dnd? */
	int hascallerid;
	int hidecallerid;
	int dtmfinband;
	int amaflags;
	int type;
	int group;
	int iseq; /* Not used? */
	int lastout; /* tracking this on the subchannels.  Is it needed here? */
	int needdestroy; /* Not used? */
	int capability;
	int nonCodecCapability;
    int onhooktime;
    int msgstate; /* voicemail message state */
    int immediate;
    int hookstate;
    int adsi;
	struct ast_dsp *dsp; /* XXX Should there be a dsp/subchannel? XXX */
    /* owner is tracked on the subchannels, and the *sub indicates whos in charge */
	/* struct ast_channel *owner; */
	/* struct ast_rtp *rtp; */
	/* struct sockaddr_in tmpdest; */
    /* message go the the endpoint and not the channel so they stay here */
	struct mgcp_endpoint *next;
	struct mgcp_gateway *parent;
};

static struct mgcp_gateway {
	/* A gateway containing one or more endpoints */
	char name[80];
	struct sockaddr_in addr;
	struct sockaddr_in defaddr;
	struct in_addr ourip;
	int dynamic;
	int expire;		/* XXX Should we ever expire dynamic registrations? XXX */
	struct mgcp_endpoint *endpoints;
	struct ast_ha *ha;
	struct mgcp_gateway *next;
} *gateways;

static ast_mutex_t gatelock  = AST_MUTEX_INITIALIZER;

static int mgcpsock  = -1;

static struct sockaddr_in bindaddr;

static struct ast_frame  *mgcp_read(struct ast_channel *ast);
static int transmit_response(struct mgcp_subchannel *sub, char *msg, struct mgcp_request *req, char *msgrest);
static int transmit_notify_request(struct mgcp_subchannel *sub, char *tone);
static int transmit_modify_request(struct mgcp_subchannel *sub);
static int transmit_notify_request_with_callerid(struct mgcp_subchannel *sub, char *tone, char *callerid);
static int transmit_modify_with_sdp(struct mgcp_subchannel *sub, struct ast_rtp *rtp);
static int transmit_connection_del(struct mgcp_subchannel *sub);
static int transmit_audit_endpoint(struct mgcp_endpoint *p);
static void start_rtp(struct mgcp_subchannel *sub);

static int has_voicemail(struct mgcp_endpoint *p)
{
	return ast_app_has_voicemail(p->mailbox);
}

static int unalloc_sub(struct mgcp_subchannel *sub)
{
    struct mgcp_endpoint *p = sub->parent;
	if (p->sub == sub) {
		ast_log(LOG_WARNING, "Trying to unalloc the real channel %s@%s?!?\n", p->name, p->parent->name);
		return -1;
	}
	ast_log(LOG_DEBUG, "Released sub %d of channel %s@%s\n", sub->id, p->name, p->parent->name);

	sub->owner = NULL;
	if (strlen(sub->cxident)) {
		transmit_connection_del(sub);
    }
    sub->cxident[0] = '\0';
    sub->callid[0] = '\0';
    sub->cxmode = MGCP_CX_INACTIVE;
	sub->outgoing = 0;
	sub->alreadygone = 0;
	memset(&sub->tmpdest, 0, sizeof(sub->tmpdest));
	if (sub->rtp) {
		ast_rtp_destroy(sub->rtp);
		sub->rtp = NULL;
	}
	return 0;
}

static int __mgcp_xmit(struct mgcp_subchannel *sub, char *data, int len)
{
    struct mgcp_endpoint *p = sub->parent;
	int res;
	if (p->parent->addr.sin_addr.s_addr)
	    res=sendto(mgcpsock, data, len, 0, (struct sockaddr *)&p->parent->addr, sizeof(struct sockaddr_in));
	else 
	    res=sendto(mgcpsock, data, len, 0, (struct sockaddr *)&p->parent->defaddr, sizeof(struct sockaddr_in));
	if (res != len) {
		ast_log(LOG_WARNING, "mgcp_xmit returned %d: %s\n", res, strerror(errno));
	}
	return res;
}

static int send_response(struct mgcp_subchannel *sub, struct mgcp_request *req)
{
    struct mgcp_endpoint *p = sub->parent;
	int res;
	if (mgcpdebug) {
		ast_verbose("Transmitting:\n%s\n to %s:%d\n", req->data, inet_ntoa(p->parent->addr.sin_addr), ntohs(p->parent->addr.sin_port));
    }
	res = __mgcp_xmit(sub, req->data, req->len);
	if (res > 0)
		res = 0;
	return res;
}

static void dump_queue(struct mgcp_endpoint *p)
{
	struct mgcp_message *cur;
    struct mgcp_subchannel *sub = p->sub;
    do {
        while(sub->msgs) {
            cur = sub->msgs;
            sub->msgs = sub->msgs->next;
            ast_log(LOG_NOTICE, "Removing message from %s@%s-%d tansaction %d\n", p->name, p->parent->name, sub->id, cur->seqno);
            free(cur);
        }
        sub->messagepending = 0;
        sub->msgs = NULL;
        sub = sub->next;
    } while(sub != p->sub);
}

static int mgcp_postrequest(struct mgcp_subchannel *sub, unsigned char *data, int len, unsigned int seqno)
{
	struct mgcp_message *msg = malloc(sizeof(struct mgcp_message) + len);
	struct mgcp_message *cur;
    time_t t;
	if (!msg) {
		return -1;
    }
    time(&t);
    if (sub->messagepending && (sub->lastouttime + 20 < t)) {
        ast_log(LOG_NOTICE, "Timeout waiting for response to message:%d,  lastouttime: %ld, now: %ld.  Dumping pending queue\n",
                sub->msgs ? sub->msgs->seqno : -1, (long) sub->lastouttime, (long) t);
        dump_queue(sub->parent);
    }
	msg->seqno = seqno;
	msg->next = NULL;
	msg->len = len;
	memcpy(msg->buf, data, msg->len);
	cur = sub->msgs;
	if (cur) {
		while(cur->next)
			cur = cur->next;
		cur->next = msg;
	} else {
		sub->msgs = msg;
    }
	if (!sub->messagepending) {
		sub->messagepending = 1;
		sub->lastout = seqno;
        sub->lastouttime = t;
		__mgcp_xmit(sub, msg->buf, msg->len);
		/* XXX Should schedule retransmission XXX */
	} else
		ast_log(LOG_DEBUG, "Deferring transmission of transaction %d\n", seqno);
	return 0;
}

static int send_request(struct mgcp_subchannel *sub, struct mgcp_request *req, unsigned int seqno)
{
	int res;
    struct mgcp_endpoint *p = sub->parent;
	if (mgcpdebug) {
		ast_verbose("Posting Request:\n%s to %s:%d\n", req->data, 
                inet_ntoa(p->parent->addr.sin_addr), ntohs(p->parent->addr.sin_port));
    }
		
	res = mgcp_postrequest(sub, req->data, req->len, seqno);
	return res;
}

static int mgcp_call(struct ast_channel *ast, char *dest, int timeout)
{
	int res;
	struct mgcp_endpoint *p;
    struct mgcp_subchannel *sub;
    char *tone;

    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_call(%s)\n", ast->name);
    }
	sub = ast->pvt->pvt;
    p = sub->parent;

    switch (p->hookstate) {
        case MGCP_OFFHOOK:
            tone = "L/wt";
            break;
        case MGCP_ONHOOK:
        default:
            tone = "L/rg";
            break;
    }

	if ((ast->_state != AST_STATE_DOWN) && (ast->_state != AST_STATE_RESERVED)) {
		ast_log(LOG_WARNING, "mgcp_call called on %s, neither down nor reserved\n", ast->name);
		return -1;
	}

	res = 0;
	sub->outgoing = 1;
    sub->cxmode = MGCP_CX_RECVONLY;
	if (p->type == TYPE_LINE) {
        if (!sub->rtp) {
            start_rtp(sub);
        } else {
            transmit_modify_request(sub);
        }

        if (sub->next->owner && strlen(sub->next->cxident) && strlen(sub->next->callid)) {
            /* try to prevent a callwait from disturbing the other connection */
            sub->next->cxmode = MGCP_CX_RECVONLY;
            transmit_modify_request(sub->next);
        }

		transmit_notify_request_with_callerid(sub, tone, ast->callerid);
		ast_setstate(ast, AST_STATE_RINGING);
		ast_queue_control(ast, AST_CONTROL_RINGING, 0);

        if (sub->next->owner && strlen(sub->next->cxident) && strlen(sub->next->callid)) {
            /* Put the connection back in sendrecv */
            sub->next->cxmode = MGCP_CX_SENDRECV;
            transmit_modify_request(sub->next);
        }

	} else {
		ast_log(LOG_NOTICE, "Don't know how to dial on trunks yet\n");
		res = -1;
	}
	return res;
}

static int mgcp_hangup(struct ast_channel *ast)
{
	struct mgcp_subchannel *sub = ast->pvt->pvt;
	struct mgcp_endpoint *p = sub->parent;

    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_hangup(%s) on %s@%s\n", ast->name, p->name, p->parent->name);
    }
	if (option_debug)
		ast_log(LOG_DEBUG, "mgcp_hangup(%s)\n", ast->name);
	if (!ast->pvt->pvt) {
		ast_log(LOG_DEBUG, "Asked to hangup channel not connected\n");
		return 0;
	}
	if ((p->dtmfinband) && (p->dsp != NULL)){
	    ast_dsp_free(p->dsp);
	}
	ast_mutex_lock(&sub->lock);

	sub->owner = NULL;
	if (strlen(sub->cxident)) {
		transmit_connection_del(sub);
    }
	strcpy(sub->cxident, "");
    if ((sub == p->sub) && sub->next->owner) {
        if (p->hookstate == MGCP_OFFHOOK) {
            if (sub->next->owner && sub->next->owner->bridge) {
                transmit_notify_request_with_callerid(p->sub, "L/wt", sub->next->owner->bridge->callerid);
            }
        } else {
            /* set our other connection as the primary and swith over to it */
            p->sub = sub->next;
            p->sub->cxmode = MGCP_CX_RECVONLY;
            transmit_modify_request(p->sub);
            if (sub->next->owner && sub->next->owner->bridge) {
                transmit_notify_request_with_callerid(p->sub, "L/rg", sub->next->owner->callerid);
            }
        }

    } else if ((sub == p->sub->next) && p->hookstate == MGCP_OFFHOOK) {
        transmit_notify_request(sub, "v");
    } else if (p->hookstate == MGCP_OFFHOOK) {
        transmit_notify_request(sub, "ro");
    } else {
        transmit_notify_request(sub, "");
    }

	ast->pvt->pvt = NULL;
	sub->alreadygone = 0;
	sub->outgoing = 0;
	sub->cxmode = MGCP_CX_INACTIVE;
	strcpy(sub->callid, "");
	/* Reset temporary destination */
	memset(&sub->tmpdest, 0, sizeof(sub->tmpdest));
	if (sub->rtp) {
		ast_rtp_destroy(sub->rtp);
		sub->rtp = NULL;
	}

    if ((p->hookstate == MGCP_ONHOOK) && (!sub->next->rtp)) {
        if (has_voicemail(p)) {
            if (mgcpdebug) {
                ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_hangup(%s) on %s@%s set vmwi(+)\n", ast->name, p->name, p->parent->name);
            }
            transmit_notify_request(sub, "vmwi(+)");
        } else {
            if (mgcpdebug) {
                ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_hangup(%s) on %s@%s set vmwi(-)\n", ast->name, p->name, p->parent->name);
            }
            transmit_notify_request(sub, "vmwi(-)");
        }
    }
	ast_mutex_unlock(&sub->lock);
	return 0;
}

static int mgcp_show_endpoints(int fd, int argc, char *argv[])
{
	struct mgcp_gateway  *g;
	struct mgcp_endpoint *e;
	int hasendpoints = 0;
	if (argc != 3) 
		return RESULT_SHOWUSAGE;
	ast_mutex_lock(&gatelock);
	g = gateways;
	while(g) {
		e = g->endpoints;
		ast_cli(fd, "Gateway '%s' at %s (%s)\n", g->name, g->addr.sin_addr.s_addr ? inet_ntoa(g->addr.sin_addr) : inet_ntoa(g->defaddr.sin_addr), g->dynamic ? "Dynamic" : "Static");
		while(e) {
			ast_cli(fd, "   -- '%s@%s in '%s' is %s\n", e->name, g->name, e->context, e->sub->owner ? "active" : "idle");
			hasendpoints = 1;
			e = e->next;
		}
		if (!hasendpoints) {
			ast_cli(fd, "   << No Endpoints Defined >>     ");
		}
		g = g->next;
	}
	ast_mutex_unlock(&gatelock);
	return RESULT_SUCCESS;
}

static char show_endpoints_usage[] = 
"Usage: mgcp show endpoints\n"
"       Lists all endpoints known to the MGCP (Media Gateawy Control Protocol) subsystem.\n";

static struct ast_cli_entry  cli_show_endpoints = 
	{ { "mgcp", "show", "endpoints", NULL }, mgcp_show_endpoints, "Show defined MGCP endpoints", show_endpoints_usage };

static int mgcp_audit_endpoint(int fd, int argc, char *argv[])
{
	struct mgcp_gateway  *g;
	struct mgcp_endpoint *e;
	int found = 0;
    char *ename,*gname, *c;
	if (!mgcpdebug) {
		return RESULT_SHOWUSAGE;
    }
	if (argc != 4) 
		return RESULT_SHOWUSAGE;
    /* split the name into parts by null */
    ename = argv[3];
    gname = ename;
    while (*gname) {
        if (*gname == '@') {
            *gname = 0;
            gname++;
            break;
        }
        gname++;
    }
	if (gname[0] == '[')
		gname++;
	if ((c = strrchr(gname, ']')))
		*c = '\0';
	ast_mutex_lock(&gatelock);
	g = gateways;
	while(g) {
        if (!strcasecmp(g->name, gname)) {
            e = g->endpoints;
            while(e) {
                if (!strcasecmp(e->name, ename)) {
                    found = 1;
                    transmit_audit_endpoint(e);
                    break;
                }
                e = e->next;
            }
            if (found) {
                break;
            }
        }
        g = g->next;
	}
    if (!found) {
        ast_cli(fd, "   << Could not find endpoint >>     ");
    }
	ast_mutex_unlock(&gatelock);
	return RESULT_SUCCESS;
}

static char audit_endpoint_usage[] = 
"Usage: mgcp audit endpoint <endpointid>\n"
"       List the capabilities of an endpoint in the MGCP (Media Gateawy Control Protocol) subsystem.\n"
"       mgcp debug MUST be on to see the results of this command.\n";

static struct ast_cli_entry  cli_audit_endpoint = 
	{ { "mgcp", "audit", "endpoint", NULL }, mgcp_audit_endpoint, "Audit specified MGCP endpoint", audit_endpoint_usage };

static int mgcp_answer(struct ast_channel *ast)
{
	int res = 0;
	struct mgcp_subchannel *sub = ast->pvt->pvt;
	struct mgcp_endpoint *p = sub->parent;
    sub->cxmode = MGCP_CX_SENDRECV;
    if (!sub->rtp) {
        start_rtp(sub);
    } else {
        transmit_modify_request(sub);
    }
    ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_answer(%s) on %s@%s-%d\n", ast->name, p->name, p->parent->name, sub->id);
	if (ast->_state != AST_STATE_UP) {
		ast_setstate(ast, AST_STATE_UP);
		if (option_debug)
			ast_log(LOG_DEBUG, "mgcp_answer(%s)\n", ast->name);
		transmit_notify_request(sub, "");
		transmit_modify_request(sub);
	}
	return res;
}

static struct ast_frame *mgcp_rtp_read(struct mgcp_subchannel *sub)
{
	/* Retrieve audio/etc from channel.  Assumes sub->lock is already held. */
	struct ast_frame *f;
	f = ast_rtp_read(sub->rtp);
	if (sub->owner) {
		/* We already hold the channel lock */
		if (f->frametype == AST_FRAME_VOICE) {
			if (f->subclass != sub->owner->nativeformats) {
				ast_log(LOG_DEBUG, "Oooh, format changed to %d\n", f->subclass);
				sub->owner->nativeformats = f->subclass;
				ast_set_read_format(sub->owner, sub->owner->readformat);
				ast_set_write_format(sub->owner, sub->owner->writeformat);
			}
            /* Courtesy fearnor aka alex@pilosoft.com */
            if (sub->parent->dtmfinband) {
#if 0
                ast_log(LOG_NOTICE, "MGCP ast_dsp_process\n");
#endif
                f = ast_dsp_process(sub->owner, sub->parent->dsp, f, 0);
            }
		}
	}
	return f;
}


static struct ast_frame  *mgcp_read(struct ast_channel *ast)
{
	struct ast_frame *fr;
	struct mgcp_subchannel *sub = ast->pvt->pvt;
	ast_mutex_lock(&sub->lock);
	fr = mgcp_rtp_read(sub);
	ast_mutex_unlock(&sub->lock);
	return fr;
}

static int mgcp_write(struct ast_channel *ast, struct ast_frame *frame)
{
	struct mgcp_subchannel *sub = ast->pvt->pvt;
	int res = 0;
	if (frame->frametype != AST_FRAME_VOICE) {
		if (frame->frametype == AST_FRAME_IMAGE)
			return 0;
		else {
			ast_log(LOG_WARNING, "Can't send %d type frames with MGCP write\n", frame->frametype);
			return 0;
		}
	} else {
		if (!(frame->subclass & ast->nativeformats)) {
			ast_log(LOG_WARNING, "Asked to transmit frame type %d, while native formats is %d (read/write = %d/%d)\n",
				frame->subclass, ast->nativeformats, ast->readformat, ast->writeformat);
			return -1;
		}
	}
	if (sub) {
		ast_mutex_lock(&sub->lock);
		if (sub->rtp) {
			res =  ast_rtp_write(sub->rtp, frame);
		}
		ast_mutex_unlock(&sub->lock);
	}
	return res;
}

static int mgcp_fixup(struct ast_channel *oldchan, struct ast_channel *newchan)
{
	struct mgcp_subchannel *sub = newchan->pvt->pvt;
    ast_log(LOG_NOTICE, "mgcp_fixup(%s, %s)\n", oldchan->name, newchan->name);
	if (sub->owner != oldchan) {
		ast_log(LOG_WARNING, "old channel wasn't %p but was %p\n", oldchan, sub->owner);
		return -1;
	}
	sub->owner = newchan;
	return 0;
}

static int mgcp_senddigit(struct ast_channel *ast, char digit)
{
	struct mgcp_subchannel *sub = ast->pvt->pvt;
	char tmp[2];
	tmp[0] = digit;
	tmp[1] = '\0';
	transmit_notify_request(sub, tmp);
	return -1;
}

static char *control2str(int ind) {
    switch (ind) {
        case AST_CONTROL_HANGUP:
            return "Other end has hungup";
        case AST_CONTROL_RING:
            return "Local ring";
        case AST_CONTROL_RINGING:
            return "Remote end is ringing";
        case AST_CONTROL_ANSWER:
            return "Remote end has answered";
        case AST_CONTROL_BUSY:
            return "Remote end is busy";
        case AST_CONTROL_TAKEOFFHOOK:
            return "Make it go off hook";
        case AST_CONTROL_OFFHOOK:
            return "Line is off hook";
        case AST_CONTROL_CONGESTION:
            return "Congestion (circuits busy)";
        case AST_CONTROL_FLASH:
            return "Flash hook";
        case AST_CONTROL_WINK:
            return "Wink";
        case AST_CONTROL_OPTION:
            return "Set a low-level option";
        case AST_CONTROL_RADIO_KEY:
            return "Key Radio";
        case AST_CONTROL_RADIO_UNKEY:
            return "Un-Key Radio";
    }
    return "UNKNOWN";
}


static int mgcp_indicate(struct ast_channel *ast, int ind)
{
	struct mgcp_subchannel *sub = ast->pvt->pvt;
    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP asked to indicate %d '%s' condition on channel %s\n", ind, control2str(ind), ast->name);
    }
	switch(ind) {
	case AST_CONTROL_RINGING:
		transmit_notify_request(sub, "rt");
		break;
	case AST_CONTROL_BUSY:
		transmit_notify_request(sub, "bz");
		break;
	case AST_CONTROL_CONGESTION:
		transmit_notify_request(sub, "cg");
		break;
	case -1:
		transmit_notify_request(sub, "");
		break;		
	default:
		ast_log(LOG_WARNING, "Don't know how to indicate condition %d\n", ind);
		return -1;
	}
	return 0;
}

static struct ast_channel *mgcp_new(struct mgcp_subchannel *sub, int state)
{
	struct ast_channel *tmp;
    struct mgcp_endpoint *i = sub->parent;
	int fmt;
    i = sub->parent;
	tmp = ast_channel_alloc(1);
	if (tmp) {
		tmp->nativeformats = i->capability;
		if (!tmp->nativeformats)
			tmp->nativeformats = capability;
		fmt = ast_best_codec(tmp->nativeformats);
		snprintf(tmp->name, sizeof(tmp->name), "MGCP/%s@%s-%d", i->name, i->parent->name, sub->id);
		if (sub->rtp)
			tmp->fds[0] = ast_rtp_fd(sub->rtp);
		tmp->type = type;
		if (i->dtmfinband) {
		    i->dsp = ast_dsp_new();
		    ast_dsp_set_features(i->dsp,DSP_FEATURE_DTMF_DETECT);
		} else {
		    i->dsp = NULL;
		}
		ast_setstate(tmp, state);
		if (state == AST_STATE_RING)
			tmp->rings = 1;
		tmp->writeformat = fmt;
		tmp->pvt->rawwriteformat = fmt;
		tmp->readformat = fmt;
		tmp->pvt->rawreadformat = fmt;
		tmp->pvt->pvt = sub;
		tmp->pvt->call = mgcp_call;
		tmp->pvt->hangup = mgcp_hangup;
		tmp->pvt->answer = mgcp_answer;
		tmp->pvt->read = mgcp_read;
		tmp->pvt->write = mgcp_write;
		tmp->pvt->indicate = mgcp_indicate;
		tmp->pvt->fixup = mgcp_fixup;
		tmp->pvt->send_digit = mgcp_senddigit;
		tmp->pvt->bridge = ast_rtp_bridge;
		if (strlen(i->language))
			strncpy(tmp->language, i->language, sizeof(tmp->language)-1);
		if (strlen(i->accountcode))
			strncpy(tmp->accountcode, i->accountcode, sizeof(tmp->accountcode)-1);
		if (i->amaflags)
			tmp->amaflags = i->amaflags;
		sub->owner = tmp;
		ast_mutex_lock(&usecnt_lock);
		usecnt++;
		ast_mutex_unlock(&usecnt_lock);
		ast_update_use_count();
		tmp->callgroup = i->callgroup;
		tmp->pickupgroup = i->pickupgroup;
		strncpy(tmp->call_forward, i->call_forward, sizeof(tmp->call_forward));
		strncpy(tmp->context, i->context, sizeof(tmp->context)-1);
		strncpy(tmp->exten, i->exten, sizeof(tmp->exten)-1);
		if (strlen(i->callerid))
			tmp->callerid = strdup(i->callerid);
		if (!i->adsi)
			tmp->adsicpe = AST_ADSI_UNAVAILABLE;
		tmp->priority = 1;
		if (state != AST_STATE_DOWN) {
			if (ast_pbx_start(tmp)) {
				ast_log(LOG_WARNING, "Unable to start PBX on %s\n", tmp->name);
				ast_hangup(tmp);
				tmp = NULL;
			}
		}
        ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_new(%s) created in state: %s\n", tmp->name, ast_state2str(state));
	} else {
		ast_log(LOG_WARNING, "Unable to allocate channel structure\n");
    }
	return tmp;
}

static char* get_sdp_by_line(char* line, char *name, int nameLen) {
  if (strncasecmp(line, name, nameLen) == 0 && line[nameLen] == '=') {
    char* r = line + nameLen + 1;
    while (*r && (*r < 33)) ++r;
    return r;
  }

  return "";
}

static char *get_sdp(struct mgcp_request *req, char *name) {
  int x;
  int len = strlen(name);
  char *r;

  for (x=0; x<req->lines; x++) {
    r = get_sdp_by_line(req->line[x], name, len);
    if (r[0] != '\0') return r;
  }
  return "";
}

static void sdpLineNum_iterator_init(int* iterator) {
  *iterator = 0;
}

static char* get_sdp_iterate(int* iterator,
			     struct mgcp_request *req, char *name) {
  int len = strlen(name);
  char *r;
  while (*iterator < req->lines) {
    r = get_sdp_by_line(req->line[(*iterator)++], name, len);
    if (r[0] != '\0') return r;
  }
  return "";
}

static char *__get_header(struct mgcp_request *req, char *name, int *start)
{
	int x;
	int len = strlen(name);
	char *r;
	for (x=*start;x<req->headers;x++) {
		if (!strncasecmp(req->header[x], name, len) && 
				(req->header[x][len] == ':')) {
					r = req->header[x] + len + 1;
					while(*r && (*r < 33))
							r++;
					*start = x+1;
					return r;
		}
	}
	/* Don't return NULL, so get_header is always a valid pointer */
	return "";
}

static char *get_header(struct mgcp_request *req, char *name)
{
	int start = 0;
	return __get_header(req, name, &start);
}

#if 0
static int rtpready(struct ast_rtp *rtp, struct ast_frame *f, void *data)
{
	/* Just deliver the audio directly */
	struct mgcp_endpoint *p = data;
	ast_mutex_lock(&p->lock);
	if (p->owner) {
		/* Generally, you lock in the order channel lock, followed by private
		   lock.  Since here we are doing the reverse, there is the possibility
		   of deadlock.  As a result, in the case of a deadlock, we simply fail out
		   here. */
		if (!ast_mutex_trylock(&p->owner->lock)) {
			if (f->frametype == AST_FRAME_VOICE) {
				if (f->subclass != p->owner->nativeformats) {
					ast_log(LOG_DEBUG, "Oooh, format changed to %d\n", f->subclass);
					p->owner->nativeformats = f->subclass;
					ast_set_read_format(p->owner, p->owner->readformat);
					ast_set_write_format(p->owner, p->owner->writeformat);
				}
				if (p->dtmfinband) {
				    f = ast_dsp_process(p->owner,p->dsp,f,0);
				}
			}
			ast_queue_frame(p->owner, f, 0);
			ast_mutex_unlock(&p->owner->lock);
		}
	}
	ast_mutex_unlock(&p->lock);
	return 0;
}
#endif

static struct mgcp_subchannel *find_subchannel(char *name, int msgid, struct sockaddr_in *sin)
{
	struct mgcp_endpoint *p = NULL;
	struct mgcp_subchannel *sub = NULL;
	struct mgcp_gateway *g;
	char tmp[256] = "";
	char *at = NULL, *c;
    int found = 0;
	if (name) {
		strncpy(tmp, name, sizeof(tmp) - 1);
		at = strchr(tmp, '@');
		if (!at) {
			ast_log(LOG_NOTICE, "Endpoint '%s' has no at sign!\n", name);
			return NULL;
		}
		*at = '\0';
		at++;
	}
	ast_mutex_lock(&gatelock);
	if (at && (at[0] == '[')) {
		at++;
		c = strrchr(at, ']');
		if (c)
			*c = '\0';
	}
	g = gateways;
	while(g) {
		if ((!name || !strcasecmp(g->name, at)) && 
		    (sin || g->addr.sin_addr.s_addr || g->defaddr.sin_addr.s_addr)) {
			/* Found the gateway.  If it's dynamic, save it's address -- now for the endpoint */
			if (sin && g->dynamic && name) {
				if ((g->addr.sin_addr.s_addr != sin->sin_addr.s_addr) ||
					(g->addr.sin_port != sin->sin_port)) {
					memcpy(&g->addr, sin, sizeof(g->addr));
					if (ast_ouraddrfor(&g->addr.sin_addr, &g->ourip))
						memcpy(&g->ourip, &__ourip, sizeof(g->ourip));
					if (option_verbose > 2)
						ast_verbose(VERBOSE_PREFIX_3 "Registered MGCP gateway '%s' at %s port %d\n", g->name, inet_ntoa(g->addr.sin_addr), ntohs(g->addr.sin_port));
				}
			}
			p = g->endpoints;
			while(p) {
				if (option_debug)
	                ast_log(LOG_DEBUG, "Searching on %s@%s for subchannel\n", p->name, g->name);
                if (msgid) {
                    sub = p->sub;
                    do {
						if (option_debug)
    	                    ast_log(LOG_DEBUG, "Searching on %s@%s-%d for subchannel with lastout: %d\n", p->name, g->name, sub->id, msgid);
                        if (sub->lastout == msgid) {
							if (option_debug)
            	                ast_log(LOG_DEBUG, "Found subchannel sub%d to handle request %d sub->lastout: %d\n", sub->id, msgid, sub->lastout);
                            found = 1;
                            break;
                        }
                        sub = sub->next;
                    } while (sub != p->sub);
                    if (found) {
                        break;
                    }
                } else if (name && !strcasecmp(p->name, tmp)) {
                    ast_log(LOG_DEBUG, "Coundn't determine subchannel, assuming current master %s@%s-%d\n", 
                            p->name, g->name, p->sub->id);
                    sub = p->sub;
                    found = 1;
                    break;
                }
                p = p->next;
			}
			if (sub && found) {
				break;
            }
		}
		g = g->next;
	}
	ast_mutex_unlock(&gatelock);
	if (!sub) {
		if (name) {
			if (g)
				ast_log(LOG_NOTICE, "Endpoint '%s' not found on gateway '%s'\n", tmp,at);
			else
				ast_log(LOG_NOTICE, "Gateway '%s' (and thus its endpoint '%s') does not exist\n", at, tmp);
		} 
	}
	return sub;
}

static void parse(struct mgcp_request *req)
{
	/* Divide fields by NULL's */
	char *c;
	int f = 0;
	c = req->data;

	/* First header starts immediately */
	req->header[f] = c;
	while(*c) {
		if (*c == '\n') {
			/* We've got a new header */
			*c = 0;

#if 0
			printf("Header: %s (%d)\n", req->header[f], strlen(req->header[f]));
#endif			
			if (!strlen(req->header[f])) {
				/* Line by itself means we're now in content */
				c++;
				break;
			}
			if (f >= MGCP_MAX_HEADERS - 1) {
				ast_log(LOG_WARNING, "Too many MGCP headers...\n");
			} else
				f++;
			req->header[f] = c + 1;
		} else if (*c == '\r') {
			/* Ignore but eliminate \r's */
			*c = 0;
		}
		c++;
	}
	/* Check for last header */
	if (strlen(req->header[f])) 
		f++;
	req->headers = f;
	/* Now we process any mime content */
	f = 0;
	req->line[f] = c;
	while(*c) {
		if (*c == '\n') {
			/* We've got a new line */
			*c = 0;
#if 0
			printf("Line: %s (%d)\n", req->line[f], strlen(req->line[f]));
#endif			
			if (f >= MGCP_MAX_LINES - 1) {
				ast_log(LOG_WARNING, "Too many SDP lines...\n");
			} else
				f++;
			req->line[f] = c + 1;
		} else if (*c == '\r') {
			/* Ignore and eliminate \r's */
			*c = 0;
		}
		c++;
	}
	/* Check for last line */
	if (strlen(req->line[f])) 
		f++;
	req->lines = f;
	/* Parse up the initial header */
	c = req->header[0];
	while(*c && *c < 33) c++;
	/* First the verb */
	req->verb = c;
	while(*c && (*c > 32)) c++;
	if (*c) {
		*c = '\0';
		c++;
		while(*c && (*c < 33)) c++;
		req->identifier = c;
		while(*c && (*c > 32)) c++;
		if (*c) {
			*c = '\0';
			c++;
			while(*c && (*c < 33)) c++;
			req->endpoint = c;
			while(*c && (*c > 32)) c++;
			if (*c) {
				*c = '\0';
				c++;
				while(*c && (*c < 33)) c++;
				req->version = c;
				while(*c && (*c > 32)) c++;
				while(*c && (*c < 33)) c++;
				while(*c && (*c > 32)) c++;
				*c = '\0';
			}
		}
	}
		
	if (mgcpdebug) {
		ast_verbose("Verb: '%s', Identifier: '%s', Endpoint: '%s', Version: '%s'\n",
                    req->verb, req->identifier, req->endpoint, req->version);
		ast_verbose("%d headers, %d lines\n", req->headers, req->lines);
	}
	if (*c) 
		ast_log(LOG_WARNING, "Odd content, extra stuff left over ('%s')\n", c);
}

static int process_sdp(struct mgcp_subchannel *sub, struct mgcp_request *req)
{
	char *m;
	char *c;
	char *a;
	char host[258];
	int len;
	int portno;
	int peercapability, peerNonCodecCapability;
	struct sockaddr_in sin;
	char *codecs;
	struct hostent *hp;
	int codec;
	int iterator;
    struct mgcp_endpoint *p = sub->parent;

	/* Get codec and RTP info from SDP */
	m = get_sdp(req, "m");
	c = get_sdp(req, "c");
	if (!strlen(m) || !strlen(c)) {
		ast_log(LOG_WARNING, "Insufficient information for SDP (m = '%s', c = '%s')\n", m, c);
		return -1;
	}
	if (sscanf(c, "IN IP4 %256s", host) != 1) {
		ast_log(LOG_WARNING, "Invalid host in c= line, '%s'\n", c);
		return -1;
	}
	/* XXX This could block for a long time, and block the main thread! XXX */
	hp = gethostbyname(host);
	if (!hp) {
		ast_log(LOG_WARNING, "Unable to lookup host in c= line, '%s'\n", c);
		return -1;
	}
	if (sscanf(m, "audio %d RTP/AVP %n", &portno, &len) != 1) {
		ast_log(LOG_WARNING, "Unable to determine port number for RTP in '%s'\n", m); 
		return -1;
	}
	sin.sin_family = AF_INET;
	memcpy(&sin.sin_addr, hp->h_addr, sizeof(sin.sin_addr));
	sin.sin_port = htons(portno);
	ast_rtp_set_peer(sub->rtp, &sin);
#if 0
	printf("Peer RTP is at port %s:%d\n", inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
#endif	
	// Scan through the RTP payload types specified in a "m=" line:
    ast_rtp_pt_clear(sub->rtp);
	codecs = m + len;
	while(strlen(codecs)) {
		if (sscanf(codecs, "%d %n", &codec, &len) != 1) {
			ast_log(LOG_WARNING, "Error in codec string '%s'\n", codecs);
			return -1;
		}
		ast_rtp_set_m_type(sub->rtp, codec);
		codecs += len;
	}

        // Next, scan through each "a=rtpmap:" line, noting each
        // specified RTP payload type (with corresponding MIME subtype):
        sdpLineNum_iterator_init(&iterator);
        while ((a = get_sdp_iterate(&iterator, req, "a"))[0] != '\0') {
          char* mimeSubtype = strdup(a); // ensures we have enough space
          if (sscanf(a, "rtpmap: %u %[^/]/", &codec, mimeSubtype) != 2) continue;
          // Note: should really look at the 'freq' and '#chans' params too
          ast_rtp_set_rtpmap_type(sub->rtp, codec, "audio", mimeSubtype);
	  free(mimeSubtype);
        }

        // Now gather all of the codecs that were asked for:
        ast_rtp_get_current_formats(sub->rtp,
                                &peercapability, &peerNonCodecCapability);
	p->capability = capability & peercapability;
	if (mgcpdebug) {
		ast_verbose("Capabilities: us - %d, them - %d, combined - %d\n",
                    capability, peercapability, p->capability);
		ast_verbose("Non-codec capabilities: us - %d, them - %d, combined - %d\n",
                    nonCodecCapability, peerNonCodecCapability, p->nonCodecCapability);
	}
	if (!p->capability) {
		ast_log(LOG_WARNING, "No compatible codecs!\n");
		return -1;
	}
	return 0;
	
}

static int add_header(struct mgcp_request *req, char *var, char *value)
{
	if (req->len >= sizeof(req->data) - 4) {
		ast_log(LOG_WARNING, "Out of space, can't add anymore\n");
		return -1;
	}
	if (req->lines) {
		ast_log(LOG_WARNING, "Can't add more headers when lines have been added\n");
		return -1;
	}
	req->header[req->headers] = req->data + req->len;
	snprintf(req->header[req->headers], sizeof(req->data) - req->len, "%s: %s\r\n", var, value);
	req->len += strlen(req->header[req->headers]);
	if (req->headers < MGCP_MAX_HEADERS)
		req->headers++;
	else {
		ast_log(LOG_WARNING, "Out of header space\n");
		return -1;
	}
	return 0;	
}

static int add_line(struct mgcp_request *req, char *line)
{
	if (req->len >= sizeof(req->data) - 4) {
		ast_log(LOG_WARNING, "Out of space, can't add anymore\n");
		return -1;
	}
	if (!req->lines) {
		/* Add extra empty return */
		snprintf(req->data + req->len, sizeof(req->data) - req->len, "\r\n");
		req->len += strlen(req->data + req->len);
	}
	req->line[req->lines] = req->data + req->len;
	snprintf(req->line[req->lines], sizeof(req->data) - req->len, "%s", line);
	req->len += strlen(req->line[req->lines]);
	if (req->lines < MGCP_MAX_LINES)
		req->lines++;
	else {
		ast_log(LOG_WARNING, "Out of line space\n");
		return -1;
	}
	return 0;	
}

static int init_resp(struct mgcp_request *req, char *resp, struct mgcp_request *orig, char *resprest)
{
	/* Initialize a response */
	if (req->headers || req->len) {
		ast_log(LOG_WARNING, "Request already initialized?!?\n");
		return -1;
	}
	req->header[req->headers] = req->data + req->len;
	snprintf(req->header[req->headers], sizeof(req->data) - req->len, "%s %s %s\r\n", resp, orig->identifier, resprest);
	req->len += strlen(req->header[req->headers]);
	if (req->headers < MGCP_MAX_HEADERS)
		req->headers++;
	else
		ast_log(LOG_WARNING, "Out of header space\n");
	return 0;
}

static int init_req(struct mgcp_endpoint *p, struct mgcp_request *req, char *verb)
{
	/* Initialize a response */
	if (req->headers || req->len) {
		ast_log(LOG_WARNING, "Request already initialized?!?\n");
		return -1;
	}
	req->header[req->headers] = req->data + req->len;
	snprintf(req->header[req->headers], sizeof(req->data) - req->len, "%s %d %s@%s MGCP 1.0\r\n", verb, oseq, p->name, p->parent->name);
	req->len += strlen(req->header[req->headers]);
	if (req->headers < MGCP_MAX_HEADERS)
		req->headers++;
	else
		ast_log(LOG_WARNING, "Out of header space\n");
	return 0;
}


static int respprep(struct mgcp_request *resp, struct mgcp_endpoint *p, char *msg, struct mgcp_request *req, char *msgrest)
{
	memset(resp, 0, sizeof(*resp));
	init_resp(resp, msg, req, msgrest);
	return 0;
}

static int reqprep(struct mgcp_request *req, struct mgcp_endpoint *p, char *verb)
{
	memset(req, 0, sizeof(struct mgcp_request));
	oseq++;
	init_req(p, req, verb);
	return 0;
}

static int transmit_response(struct mgcp_subchannel *sub, char *msg, struct mgcp_request *req, char *msgrest)
{
	struct mgcp_request resp;
    struct mgcp_endpoint *p = sub->parent;
	respprep(&resp, p, msg, req, msgrest);
	return send_response(sub, &resp);
}


static int add_sdp(struct mgcp_request *resp, struct mgcp_subchannel *sub, struct ast_rtp *rtp)
{
	int len;
	int codec;
	char costr[80];
	struct sockaddr_in sin;
	char v[256];
	char s[256];
	char o[256];
	char c[256];
	char t[256];
	char m[256];
	char a[1024] = "";
	int x;
	struct sockaddr_in dest;
    struct mgcp_endpoint *p = sub->parent;
	/* XXX We break with the "recommendation" and send our IP, in order that our
	       peer doesn't have to gethostbyname() us XXX */
	len = 0;
	if (!sub->rtp) {
		ast_log(LOG_WARNING, "No way to add SDP without an RTP structure\n");
		return -1;
	}
	ast_rtp_get_us(sub->rtp, &sin);
	if (rtp) {
		ast_rtp_get_peer(rtp, &dest);
	} else {
		if (sub->tmpdest.sin_addr.s_addr) {
			dest.sin_addr = sub->tmpdest.sin_addr;
			dest.sin_port = sub->tmpdest.sin_port;
			/* Reset temporary destination */
			memset(&sub->tmpdest, 0, sizeof(sub->tmpdest));
		} else {
			dest.sin_addr = p->parent->ourip;
			dest.sin_port = sin.sin_port;
		}
	}
	if (mgcpdebug) {
		ast_verbose("We're at %s port %d\n", inet_ntoa(p->parent->ourip), ntohs(sin.sin_port));	
    }
	snprintf(v, sizeof(v), "v=0\r\n");
	snprintf(o, sizeof(o), "o=root %d %d IN IP4 %s\r\n", getpid(), getpid(), inet_ntoa(dest.sin_addr));
	snprintf(s, sizeof(s), "s=session\r\n");
	snprintf(c, sizeof(c), "c=IN IP4 %s\r\n", inet_ntoa(dest.sin_addr));
	snprintf(t, sizeof(t), "t=0 0\r\n");
	snprintf(m, sizeof(m), "m=audio %d RTP/AVP", ntohs(dest.sin_port));
	for (x = 1; x <= AST_FORMAT_MAX_AUDIO; x <<= 1) {
		if (p->capability & x) {
			if (mgcpdebug) {
				ast_verbose("Answering with capability %d\n", x);
            }
			codec = ast_rtp_lookup_code(sub->rtp, 1, x);
            if (codec > -1) {
				snprintf(costr, sizeof(costr), " %d", codec);
				strcat(m, costr);
				snprintf(costr, sizeof(costr), "a=rtpmap:%d %s/8000\r\n", codec, ast_rtp_lookup_mime_subtype(1, x));
				strcat(a, costr);
			}
		}
	}
	for (x = 1; x <= AST_RTP_MAX; x <<= 1) {
        if (p->nonCodecCapability & x) {
            if (mgcpdebug) {
                ast_verbose("Answering with non-codec capability %d\n", x);
            }
            codec = ast_rtp_lookup_code(sub->rtp, 0, x);
            if (codec > -1) {
                snprintf(costr, sizeof(costr), " %d", codec);
                strcat(m, costr);
                snprintf(costr, sizeof(costr), "a=rtpmap:%d %s/8000\r\n", codec, ast_rtp_lookup_mime_subtype(0, x));
                strcat(a, costr);
                if (x == AST_RTP_DTMF) {
                  /* Indicate we support DTMF...  Not sure about 16, but MSN supports it so dang it, we will too... */
                  snprintf(costr, sizeof costr, "a=fmtp:%d 0-16\r\n", codec);
                  strcat(a, costr);
                }
            }
        }
    }
	strcat(m, "\r\n");
	len = strlen(v) + strlen(s) + strlen(o) + strlen(c) + strlen(t) + strlen(m) + strlen(a);
	snprintf(costr, sizeof(costr), "%d", len);
	add_line(resp, v);
	add_line(resp, o);
	add_line(resp, s);
	add_line(resp, c);
	add_line(resp, t);
	add_line(resp, m);
	add_line(resp, a);
	return 0;
}

static int transmit_modify_with_sdp(struct mgcp_subchannel *sub, struct ast_rtp *rtp)
{
	struct mgcp_request resp;
	char local[256];
	char tmp[80];
	int x;
    struct mgcp_endpoint *p = sub->parent;
	if (!strlen(sub->cxident) && rtp) {
		/* We don't have a CXident yet, store the destination and
		   wait a bit */
		ast_rtp_get_peer(rtp, &sub->tmpdest);
		return 0;
	}
	snprintf(local, sizeof(local), "p:20");
	for (x=1;x<= AST_FORMAT_MAX_AUDIO; x <<= 1) {
		if (p->capability & x) {
			snprintf(tmp, sizeof(tmp), ", a:%s", ast_rtp_lookup_mime_subtype(1, x));
			strcat(local, tmp);
		}
	}
	reqprep(&resp, p, "MDCX");
	add_header(&resp, "C", sub->callid);
	add_header(&resp, "L", local);
	add_header(&resp, "M", mgcp_cxmodes[sub->cxmode]);
	add_header(&resp, "X", sub->txident);
	add_header(&resp, "I", sub->cxident);
	/*add_header(&resp, "S", "");*/
	add_sdp(&resp, sub, rtp);
	return send_request(sub, &resp, oseq);
}

static int transmit_connect_with_sdp(struct mgcp_subchannel *sub, struct ast_rtp *rtp)
{
	struct mgcp_request resp;
	char local[256];
	char tmp[80];
	int x;
    struct mgcp_endpoint *p = sub->parent;

	snprintf(local, sizeof(local), "p:20");
	for (x=1;x<= AST_FORMAT_MAX_AUDIO; x <<= 1) {
		if (p->capability & x) {
			snprintf(tmp, sizeof(tmp), ", a:%s", ast_rtp_lookup_mime_subtype(1, x));
			strcat(local, tmp);
		}
	}
    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "Creating connection for %s@%s-%d in cxmode: %s callid: %s\n", 
                    p->name, p->parent->name, sub->id, mgcp_cxmodes[sub->cxmode], sub->callid);
    }
	reqprep(&resp, p, "CRCX");
	add_header(&resp, "C", sub->callid);
	add_header(&resp, "L", local);
	add_header(&resp, "M", mgcp_cxmodes[sub->cxmode]);
	add_header(&resp, "X", sub->txident);
	/*add_header(&resp, "S", "");*/
	add_sdp(&resp, sub, rtp);
	return send_request(sub, &resp, oseq);
}

static int transmit_notify_request(struct mgcp_subchannel *sub, char *tone)
{
	struct mgcp_request resp;
    struct mgcp_endpoint *p = sub->parent;

    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP Asked to indicate tone: %s on  %s@%s-%d in cxmode: %s\n", 
                    tone, p->name, p->parent->name, sub->id, mgcp_cxmodes[sub->cxmode]);
    }
	strncpy(p->curtone, tone, sizeof(p->curtone) - 1);
	reqprep(&resp, p, "RQNT");
	add_header(&resp, "X", sub->txident);
    switch (p->hookstate) {
	    case MGCP_ONHOOK:
            add_header(&resp, "R", "hd(N)");
            break;
        case MGCP_OFFHOOK:
            add_header(&resp, "R", "hu(N), hf(N), D/[0-9#*](N)");
            break;
    }
    if (strlen(tone)) {
        add_header(&resp, "S", tone);
    }
	return send_request(sub, &resp, oseq);
}

static int transmit_notify_request_with_callerid(struct mgcp_subchannel *sub, char *tone, char *callerid)
{
	struct mgcp_request resp;
	char cid[256];
	char tone2[256];
	char *l, *n;
	time_t t;
	struct tm tm;
    struct mgcp_endpoint *p = sub->parent;
	
	time(&t);
	localtime_r(&t,&tm);
	if (callerid)
		strncpy(cid, callerid, sizeof(cid) - 1);
	else
		strcpy(cid, "");
	ast_callerid_parse(cid, &n, &l);
	if (l) {
		ast_shrink_phone_number(l);
		if (!ast_isphonenumber(l)) {
			n = l;
			l = "";
		}
	} 
	if (!n)
		n = "O";
	if (!l)
		l = "";

    /* Keep track of last callerid for blacklist and callreturn */
    strncpy(p->lastcallerid, l, sizeof(p->lastcallerid) - 1);

	snprintf(tone2, sizeof(tone2), "%s,L/ci(%02d/%02d/%02d/%02d,%s,%s)", tone, 
			tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, l, n);
	strncpy(p->curtone, tone, sizeof(p->curtone) - 1);
	reqprep(&resp, p, "RQNT");
	add_header(&resp, "X", sub->txident);
    switch (p->hookstate) {
	    case MGCP_ONHOOK:
            add_header(&resp, "R", "L/hd(N)");
            break;
        case MGCP_OFFHOOK:
            add_header(&resp, "R", "L/hu(N),L/hf(N),D/[0-9#*](N)");
            break;
    }
    if (strlen(tone2)) {
        add_header(&resp, "S", tone2);
    }
    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP Asked to indicate tone: %s on  %s@%s-%d in cxmode: %s\n", 
                    tone2, p->name, p->parent->name, sub->id, mgcp_cxmodes[sub->cxmode]);
    }
	return send_request(sub, &resp, oseq);
}

static int transmit_modify_request(struct mgcp_subchannel *sub)
{
	struct mgcp_request resp;
    struct mgcp_endpoint *p = sub->parent;
	if (!strlen(sub->cxident)) {
		/* We don't have a CXident yet, store the destination and
		   wait a bit */
		return 0;
	}
    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "Modified %s@%s-%d with new mode: %s on callid: %s\n", 
                    p->name, p->parent->name, sub->id, mgcp_cxmodes[sub->cxmode], sub->callid);
    }
	reqprep(&resp, p, "MDCX");
	add_header(&resp, "C", sub->callid);
	add_header(&resp, "M", mgcp_cxmodes[sub->cxmode]);
	add_header(&resp, "X", sub->txident);
	add_header(&resp, "I", sub->cxident);
    switch (sub->parent->hookstate) {
	    case MGCP_ONHOOK:
            add_header(&resp, "R", "L/hd(N)");
            break;
        case MGCP_OFFHOOK:
            add_header(&resp, "R", "L/hu(N),L/hf(N),D/[0-9#*](N)");
            break;
    }
	return send_request(sub, &resp, oseq);
}


static int transmit_audit_endpoint(struct mgcp_endpoint *p)
{
	struct mgcp_request resp;
	reqprep(&resp, p, "AUEP");
	add_header(&resp, "F", "A,R,D,S,X,N,I,T,O,ES,VS,E,MD,M");
	return send_request(p->sub, &resp, oseq);
}

static int transmit_connection_del(struct mgcp_subchannel *sub)
{
    struct mgcp_endpoint *p = sub->parent;
	struct mgcp_request resp;
    if (mgcpdebug) {
        ast_verbose(VERBOSE_PREFIX_3 "Delete connection %s %s@%s-%d with new mode: %s on callid: %s\n", 
                    sub->cxident, p->name, p->parent->name, sub->id, mgcp_cxmodes[sub->cxmode], sub->callid);
    }
	reqprep(&resp, p, "DLCX");
	add_header(&resp, "C", sub->callid);
	add_header(&resp, "X", sub->txident);
	add_header(&resp, "I", sub->cxident);
	return send_request(sub, &resp, oseq);
}

static void handle_response(struct mgcp_subchannel *sub, int result, int ident)
{
	struct mgcp_message *cur;
    struct mgcp_endpoint *p = sub->parent;
    time_t t;
    time(&t);
#if 0
    ast_verbose(VERBOSE_PREFIX_3 "Got response back on sub%d for transaction %d our last send was %d)\n", 
            sub->id, ident, sub->msgs ? sub->msgs->seqno : -1);
    ast_verbose(VERBOSE_PREFIX_3 "Here's out outher chan sub%d for transaction %d our last send was %d)\n", 
            sub->next->id, ident, sub->next->msgs ? sub->next->msgs->seqno : -1);
#endif
	if (sub->msgs && (sub->msgs->seqno == ident)) {
		ast_log(LOG_DEBUG, "Got response back on tansaction %d\n", ident);
		cur = sub->msgs;
		sub->msgs = sub->msgs->next;
        ast_log(LOG_DEBUG, "Removing message from %s@%s-%d tansaction %d\n", sub->parent->name, sub->parent->parent->name, sub->id, ident);
		free(cur);
		if (sub->msgs) {
			/* Send next pending message if appropriate */
			sub->messagepending = 1;
			sub->lastout = sub->msgs->seqno;
            sub->lastouttime = t;
			__mgcp_xmit(sub, sub->msgs->buf, sub->msgs->len);
			/* XXX Should schedule retransmission XXX */
		} else {
			sub->messagepending = 0;
        }
	} else {
        ast_log(LOG_NOTICE, "Got response back on %s@%s-%d for transaction %d we aren't sending? (current = %d)\n", 
                sub->parent->name, sub->parent->parent->name, sub->id, ident, sub->msgs ? sub->msgs->seqno : -1);
	}
	if ((result >= 400) && (result <= 499)) {
        switch (result) {
            case 401:
                p->hookstate = MGCP_OFFHOOK;
                break;
            case 402:
                p->hookstate = MGCP_ONHOOK;
                break;
        }
        ast_log(LOG_NOTICE, "Terminating on result %d from %s@%s-%d\n", result, p->name, p->parent->name, sub->id);
        if (sub->owner)
            ast_softhangup(sub->owner, AST_SOFTHANGUP_DEV);
	}
}

static void start_rtp(struct mgcp_subchannel *sub)
{
		ast_mutex_lock(&sub->lock);
		/* Allocate the RTP now */
		sub->rtp = ast_rtp_new(sched, io, 1, 0);
		if (sub->rtp && sub->owner)
			sub->owner->fds[0] = ast_rtp_fd(sub->rtp);
		if (sub->rtp)
			ast_rtp_setnat(sub->rtp, sub->nat);
#if 0
		ast_rtp_set_callback(p->rtp, rtpready);
		ast_rtp_set_data(p->rtp, p);
#endif		
		/* Make a call*ID */
        snprintf(sub->callid, sizeof(sub->callid), "%08x%s", rand(), sub->txident);
		/* Transmit the connection create */
		transmit_connect_with_sdp(sub, NULL);
		ast_mutex_unlock(&sub->lock);
}

static void *mgcp_ss(void *data)
{
	struct ast_channel *chan = data;
	struct mgcp_subchannel *sub = chan->pvt->pvt;
	struct mgcp_endpoint *p = sub->parent;
	char exten[AST_MAX_EXTENSION] = "";
	int len = 0;
	int timeout = firstdigittimeout;
	int res;
	int getforward=0;
    while(len < AST_MAX_EXTENSION-1) {
        res = ast_waitfordigit(chan, timeout);
        timeout = 0;
        if (res < 0) {
            ast_log(LOG_DEBUG, "waitfordigit returned < 0...\n");
            /*res = tone_zone_play_tone(p->subs[index].zfd, -1);*/
			ast_indicate(chan, -1);
            ast_hangup(chan);
            return NULL;
        } else if (res)  {
            exten[len++]=res;
            exten[len] = '\0';
        }
        if (!ast_ignore_pattern(chan->context, exten)) {
            /*res = tone_zone_play_tone(p->subs[index].zfd, -1);*/
			ast_indicate(chan, -1);
        } else {
            /* XXX Redundant?  We should already be playing dialtone */
            /*tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALTONE);*/
            transmit_notify_request(sub, "dl");
        }
        if (ast_exists_extension(chan, chan->context, exten, 1, p->callerid)) {
            if (!res || !ast_matchmore_extension(chan, chan->context, exten, 1, p->callerid)) {
                if (getforward) {
                    /* Record this as the forwarding extension */
                    strncpy(p->call_forward, exten, sizeof(p->call_forward)); 
                    if (option_verbose > 2) {
                        ast_verbose(VERBOSE_PREFIX_3 "Setting call forward to '%s' on channel %s\n", 
                                p->call_forward, chan->name);
                    }
                    /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
                    transmit_notify_request(sub, "sl");
                    if (res)
                        break;
                    usleep(500000);
                    /*res = tone_zone_play_tone(p->subs[index].zfd, -1);*/
                    ast_indicate(chan, -1);
                    sleep(1);
                    memset(exten, 0, sizeof(exten));
                    /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALTONE);*/
                    transmit_notify_request(sub, "dl");
                    len = 0;
                    getforward = 0;
                } else  {
                    /*res = tone_zone_play_tone(p->subs[index].zfd, -1);*/
                    ast_indicate(chan, -1);
                    strncpy(chan->exten, exten, sizeof(chan->exten)-1);
                    if (strlen(p->callerid)) {
                        if (!p->hidecallerid)
                            chan->callerid = strdup(p->callerid);
                        chan->ani = strdup(p->callerid);
                    }
                    ast_setstate(chan, AST_STATE_RING);
                    /*zt_enable_ec(p);*/
                    res = ast_pbx_run(chan);
                    if (res) {
                        ast_log(LOG_WARNING, "PBX exited non-zero\n");
                        /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_CONGESTION);*/
                        /*transmit_notify_request(p, "nbz", 1);*/
                        transmit_notify_request(sub, "cg");
                    }
                    return NULL;
                }
            } else {
                /* It's a match, but they just typed a digit, and there is an ambiguous match,
                   so just set the timeout to matchdigittimeout and wait some more */
                timeout = matchdigittimeout;
            }
        } else if (res == 0) {
            ast_log(LOG_DEBUG, "not enough digits (and no ambiguous match)...\n");
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_CONGESTION);*/
            transmit_notify_request(sub, "cg");
            /*zt_wait_event(p->subs[index].zfd);*/
            ast_hangup(chan);
            return NULL;
        } else if (p->callwaiting && !strcmp(exten, "*70")) {
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Disabling call waiting on %s\n", chan->name);
            }
            /* Disable call waiting if enabled */
            p->callwaiting = 0;
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            len = 0;
            memset(exten, 0, sizeof(exten));
            timeout = firstdigittimeout;
                
        } else if (!strcmp(exten,ast_pickup_ext())) {
            /* Scan all channels and see if any there
             * ringing channqels with that have call groups
             * that equal this channels pickup group  
             */
            if (ast_pickup_call(chan)) {
                ast_log(LOG_WARNING, "No call pickup possible...\n");
                /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_CONGESTION);*/
                transmit_notify_request(sub, "cg");
            }
            ast_hangup(chan);
            return NULL;
            
        } else if (!p->hidecallerid && !strcmp(exten, "*67")) {
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Disabling Caller*ID on %s\n", chan->name);
            }
            /* Disable Caller*ID if enabled */
            p->hidecallerid = 1;
            if (chan->callerid)
                free(chan->callerid);
            chan->callerid = NULL;
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            len = 0;
            memset(exten, 0, sizeof(exten));
            timeout = firstdigittimeout;
        } else if (p->callreturn && !strcmp(exten, "*69")) {
            res = 0;
            if (strlen(p->lastcallerid)) {
                res = ast_say_digit_str(chan, p->lastcallerid, "", chan->language);
            }
            if (!res)
                /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
                transmit_notify_request(sub, "sl");
            break;
        } else if (!strcmp(exten, "*78")) {
            /* Do not disturb */
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Enabled DND on channel %s\n", chan->name);
            }
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            p->dnd = 1;
            getforward = 0;
            memset(exten, 0, sizeof(exten));
            len = 0;
        } else if (!strcmp(exten, "*79")) {
            /* Do not disturb */
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Disabled DND on channel %s\n", chan->name);
            }
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            p->dnd = 0;
            getforward = 0;
            memset(exten, 0, sizeof(exten));
            len = 0;
        } else if (p->cancallforward && !strcmp(exten, "*72")) {
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            getforward = 1;
            memset(exten, 0, sizeof(exten));
            len = 0;
        } else if (p->cancallforward && !strcmp(exten, "*73")) {
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Cancelling call forwarding on channel %s\n", chan->name);
            }
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            memset(p->call_forward, 0, sizeof(p->call_forward));
            getforward = 0;
            memset(exten, 0, sizeof(exten));
            len = 0;
        } else if (!strcmp(exten, ast_parking_ext()) && 
                    sub->next->owner &&
                    sub->next->owner->bridge) {
            /* This is a three way call, the main call being a real channel, 
                and we're parking the first call. */
            ast_masq_park_call(sub->next->owner->bridge, chan, 0, NULL);
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Parking call to '%s'\n", chan->name);
            }
            break;
        } else if (strlen(p->lastcallerid) && !strcmp(exten, "*80")) {
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Blacklisting number %s\n", p->lastcallerid);
            }
            res = ast_db_put("blacklist", p->lastcallerid, "1");
            if (!res) {
                /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
                transmit_notify_request(sub, "sl");
                memset(exten, 0, sizeof(exten));
                len = 0;
            }
        } else if (p->hidecallerid && !strcmp(exten, "*82")) {
            if (option_verbose > 2) {
                ast_verbose(VERBOSE_PREFIX_3 "Enabling Caller*ID on %s\n", chan->name);
            }
            /* Enable Caller*ID if enabled */
            p->hidecallerid = 0;
            if (chan->callerid)
                free(chan->callerid);
            if (strlen(p->callerid))
                chan->callerid = strdup(p->callerid);
            /*res = tone_zone_play_tone(p->subs[index].zfd, ZT_TONE_DIALRECALL);*/
            transmit_notify_request(sub, "sl");
            len = 0;
            memset(exten, 0, sizeof(exten));
            timeout = firstdigittimeout;
        } else if (!ast_canmatch_extension(chan, chan->context, exten, 1, chan->callerid) &&
                        ((exten[0] != '*') || (strlen(exten) > 2))) {
            if (option_debug)
                ast_log(LOG_DEBUG, "Can't match %s from '%s' in context %s\n", exten, chan->callerid ? chan->callerid : "<Unknown Caller>", chan->context);
            break;
        }
        if (!timeout)
            timeout = gendigittimeout;
        if (len && !ast_ignore_pattern(chan->context, exten))
            /*tone_zone_play_tone(p->subs[index].zfd, -1);*/
			ast_indicate(chan, -1);
    }
#if 0
	for (;;) {
		res = ast_waitfordigit(chan, to);
		if (!res) {
			ast_log(LOG_DEBUG, "Timeout...\n");
			break;
		}
		if (res < 0) {
			ast_log(LOG_DEBUG, "Got hangup...\n");
            ast_hangup(chan);
			break;
		}
		exten[pos++] = res;
		if (!ast_ignore_pattern(chan->context, exten))
			ast_indicate(chan, -1);
		if (ast_matchmore_extension(chan, chan->context, exten, 1, chan->callerid)) {
			if (ast_exists_extension(chan, chan->context, exten, 1, chan->callerid)) 
				to = 3000;
			else
				to = 8000;
		} else
			break;
	}
	if (ast_exists_extension(chan, chan->context, exten, 1, chan->callerid)) {
		strncpy(chan->exten, exten, sizeof(chan->exten) - 1);
        if (!p->rtp) {
            start_rtp(p);
        }
		ast_setstate(chan, AST_STATE_RING);
		chan->rings = 1;
		if (ast_pbx_run(chan)) {
			ast_log(LOG_WARNING, "Unable to launch PBX on %s\n", chan->name);
		} else
			return NULL;
	}
#endif
	ast_hangup(chan);
	return NULL;
}

static int attempt_transfer(struct mgcp_endpoint *p)
{
    /* *************************
     * I hope this works.
     * Copied out of chan_zap
     * Cross your fingers
     * *************************/

	/* In order to transfer, we need at least one of the channels to
	   actually be in a call bridge.  We can't conference two applications
	   together (but then, why would we want to?) */
	if (p->sub->owner->bridge) {
		/* The three-way person we're about to transfer to could still be in MOH, so
		   stop if now if appropriate */
		if (p->sub->next->owner->bridge)
			ast_moh_stop(p->sub->next->owner->bridge);
		if (p->sub->owner->_state == AST_STATE_RINGING) {
			ast_indicate(p->sub->next->owner->bridge, AST_CONTROL_RINGING);
		}
		if (ast_channel_masquerade(p->sub->next->owner, p->sub->owner->bridge)) {
			ast_log(LOG_WARNING, "Unable to masquerade %s as %s\n",
					p->sub->owner->bridge->name, p->sub->next->owner->name);
			return -1;
		}
		/* Orphan the channel */
		unalloc_sub(p->sub->next);
	} else if (p->sub->next->owner->bridge) {
		if (p->sub->owner->_state == AST_STATE_RINGING) {
			ast_indicate(p->sub->next->owner->bridge, AST_CONTROL_RINGING);
		}
		ast_moh_stop(p->sub->next->owner->bridge);
		if (ast_channel_masquerade(p->sub->owner, p->sub->next->owner->bridge)) {
			ast_log(LOG_WARNING, "Unable to masquerade %s as %s\n",
					p->sub->next->owner->bridge->name, p->sub->owner->name);
			return -1;
		}
		/*swap_subs(p, SUB_THREEWAY, SUB_REAL);*/
        if (option_verbose > 2) {
            ast_verbose(VERBOSE_PREFIX_3 "Swapping %d for %d on %s@%s\n", p->sub->id, p->sub->next->id, p->name, p->parent->name);
        }
        p->sub = p->sub->next;
		unalloc_sub(p->sub->next);
		/* Tell the caller not to hangup */
		return 1;
	} else {
		ast_log(LOG_DEBUG, "Neither %s nor %s are in a bridge, nothing to transfer\n",
					p->sub->owner->name, p->sub->next->owner->name);
		p->sub->next->owner->_softhangup |= AST_SOFTHANGUP_DEV;
        if (p->sub->next->owner) {
            p->sub->next->alreadygone = 1;
            ast_queue_hangup(p->sub->next->owner, 1);
        }
	}
	return 0;
}

static void handle_hd_hf(struct mgcp_subchannel *sub, char *ev) 
{
    struct mgcp_endpoint *p = sub->parent;
    struct ast_channel *c;
	pthread_t t;
	pthread_attr_t attr;
	pthread_attr_init(&attr);
        pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);	

    /* Off hook / answer */
    if (sub->outgoing) {
        /* Answered */
        if (sub->owner) {
            if (sub->owner->bridge) {
                ast_moh_stop(sub->owner->bridge);
            }
            sub->cxmode = MGCP_CX_SENDRECV;
            if (!sub->rtp) {
                start_rtp(sub);
            } else {
                transmit_modify_request(sub);
            }
            /*transmit_notify_request(sub, "aw");*/
            transmit_notify_request(sub, "");
            ast_queue_control(sub->owner, AST_CONTROL_ANSWER, 1);
        }
    } else {
        /* Start switch */
        /*sub->cxmode = MGCP_CX_SENDRECV;*/
        if (!sub->owner) {
            if (!sub->rtp) {
                start_rtp(sub);
            } else {
                transmit_modify_request(sub);
            }
            if (p->immediate) {
                /* The channel is immediately up.  Start right away */
                transmit_notify_request(sub, "rt");
                c = mgcp_new(sub, AST_STATE_RING);
                if (!c) {
                    ast_log(LOG_WARNING, "Unable to start PBX on channel %s@%s\n", p->name, p->parent->name);
                    transmit_notify_request(sub, "cg");
                    ast_hangup(c);
                }
            } else {
                if (has_voicemail(p)) {
                    transmit_notify_request(sub, "sl");
                } else {
                    transmit_notify_request(sub, "dl");
                }
                c = mgcp_new(sub, AST_STATE_DOWN);
                if (c) {
                    if (pthread_create(&t, &attr, mgcp_ss, c)) {
                        ast_log(LOG_WARNING, "Unable to create switch thread: %s\n", strerror(errno));
                        ast_hangup(c);
                    }
                } else {
                    ast_log(LOG_WARNING, "Unable to create channel for %s@%s\n", p->name, p->parent->name);
                }
            }
        } else {
            if (p->hookstate == MGCP_OFFHOOK) {
                ast_log(LOG_WARNING, "Off hook, but alreaedy have owner on %s@%s\n", p->name, p->parent->name);
            } else {
                ast_log(LOG_WARNING, "On hook, but alreaedy have owner on %s@%s\n", p->name, p->parent->name);
                ast_log(LOG_WARNING, "If we're onhook why are we here trying to handle a hd or hf?");
            }
            if (sub->owner->bridge) {
                ast_moh_stop(sub->owner->bridge);
            }
            sub->cxmode = MGCP_CX_SENDRECV;
            if (!sub->rtp) {
                start_rtp(sub);
            } else {
                transmit_modify_request(sub);
            }
            /*transmit_notify_request(sub, "aw");*/
            transmit_notify_request(sub, "");
            /*ast_queue_control(sub->owner, AST_CONTROL_ANSWER, 1);*/
        }
    }
}

static int handle_request(struct mgcp_subchannel *sub, struct mgcp_request *req, struct sockaddr_in *sin)
{
	char *ev, *s;
	struct ast_frame f = { 0, };
    struct mgcp_endpoint *p = sub->parent;
    int res;
	if (mgcpdebug) {
		ast_verbose("Handling request '%s' on %s@%s\n", req->verb, p->name, p->parent->name);
    }
	/* Clear out potential response */
	if (!strcasecmp(req->verb, "RSIP")) {
		/* Test if this RSIP request is just a keepalive */
		if(!strcasecmp( get_header(req, "RM"), "X-keepalive")) {
			if (option_verbose > 2)
				ast_verbose(VERBOSE_PREFIX_3 "Received keepalive request from %s@%s\n", p->name, p->parent->name);
			transmit_response(sub, "200", req, "OK");
		} else {
			dump_queue(p);
			if (option_verbose > 2) {
				ast_verbose(VERBOSE_PREFIX_3 "Resetting interface %s@%s\n", p->name, p->parent->name);
		        }
			if (sub->owner) {
				ast_softhangup(sub->owner, AST_SOFTHANGUP_DEV);
	        	}
			transmit_response(sub, "200", req, "OK");
			transmit_notify_request(sub, "");
		}
	} else if (!strcasecmp(req->verb, "NTFY")) {
		/* Acknowledge and be sure we keep looking for the same things */
		transmit_response(sub, "200", req, "OK");
		/* Notified of an event */
		ev = get_header(req, "O");
		s = strchr(ev, '/');
		if (s) ev = s + 1;
        if (option_verbose > 2) {
            ast_verbose(VERBOSE_PREFIX_3 "Endpoint '%s@%s-%d' observed '%s'\n", p->name, p->parent->name, sub->id, ev);
        }
		/* Keep looking for events unless this was a hangup */
		if (strcasecmp(ev, "hu") && strcasecmp(ev, "hd")) {
			transmit_notify_request(sub, p->curtone);
        }
		if (!strcasecmp(ev, "hd")) {
            p->hookstate = MGCP_OFFHOOK;
            sub->cxmode = MGCP_CX_SENDRECV;
            handle_hd_hf(sub, ev);
		} else if (!strcasecmp(ev, "hf")) {
            /* We can assume we are offhook if we received a hookflash */
            /* First let's just do call wait and ignore threeway */
            /* We're currently in charge */
            if (p->hookstate != MGCP_OFFHOOK) {
                /* Cisco c7940 sends hf even if the phone is onhook */
                /* Thanks to point on IRC for pointing this out */
                return -1;
            }
 	    /* do not let * confrnce two down channels */  
	    if( sub->owner && sub->owner->_state == AST_STATE_DOWN && !sub->next->owner) return -1;

            if (p->callwaiting || p->transfer || p->threewaycalling) {
                if (option_verbose > 2) {
                    ast_verbose(VERBOSE_PREFIX_3 "Swapping %d for %d on %s@%s\n", p->sub->id, p->sub->next->id, p->name, p->parent->name);
                }
                p->sub = p->sub->next;

                /* transfer control to our next subchannel */
                if (!sub->next->owner) {
                    /* plave the first call on hold and start up a new call */
                    sub->cxmode = MGCP_CX_MUTE;
                    if (option_verbose > 2) {
                        ast_verbose(VERBOSE_PREFIX_3 "MGCP Muting %d on %s@%s\n", sub->id, p->name, p->parent->name);
                    }
                    transmit_modify_request(sub);
                    if (sub->owner && sub->owner->bridge) {
                        ast_moh_start(sub->owner->bridge, NULL);
                    }
                    sub->next->cxmode = MGCP_CX_RECVONLY;
                    handle_hd_hf(sub->next, ev);
                } else if (sub->owner && sub->next->owner) {
                    /* We've got two active calls lets decide whether or not to conference or just flip flop */
                    if ((!sub->outgoing) && (!sub->next->outgoing)) {
                        /* We made both calls lets conferenct */
                        if (option_verbose > 2) {
                            ast_verbose(VERBOSE_PREFIX_3 "MGCP Conferencing %d and %d on %s@%s\n", 
                                        sub->id, sub->next->id, p->name, p->parent->name);
                        }
                        sub->cxmode = MGCP_CX_CONF;
                        sub->next->cxmode = MGCP_CX_CONF;
                        if (sub->next->owner->bridge) {
                            ast_moh_stop(sub->next->owner->bridge);
                        }
                        transmit_modify_request(sub);
                        transmit_modify_request(sub->next);
                    } else {
                        /* Let's flipflop between calls */
                        /* XXX Need to check for state up ??? */
                        /* XXX Need a way to indicate the current call, or maybe the call that's waiting */
                        if (option_verbose > 2) {
                            ast_verbose(VERBOSE_PREFIX_3 "We didn't make one of the calls FLIPFLOP %d and %d on %s@%s\n", 
                                        sub->id, sub->next->id, p->name, p->parent->name);
                        }
                        sub->cxmode = MGCP_CX_MUTE;
                        if (option_verbose > 2) {
                            ast_verbose(VERBOSE_PREFIX_3 "MGCP Muting %d on %s@%s\n", sub->id, p->name, p->parent->name);
                        }
                        transmit_modify_request(sub);
                        if (sub->owner->bridge) {
                            ast_moh_start(sub->owner->bridge, NULL);
                        }
                        if (sub->next->owner->bridge) {
                            ast_moh_stop(sub->next->owner->bridge);
                        }
                        handle_hd_hf(sub->next, ev);
#if 0
                        if (sub->next->owner && (sub->next->owner->_state != AST_STATE_UP)) {
                            handle_hd_hf(sub->next, ev);
                        } else {
                            ast_verbose(VERBOSE_PREFIX_3 "MGCP Unmuting %d on %s@%s\n", sub->next->id, p->name, p->parent->name);
                            sub->next->cxmode = MGCP_CX_SENDRECV;
                            transmit_modify_request(sub->next);
                        }
#endif
                    }
                } else {
                    /* We've most likely lost one of our calls find an active call and bring it up */
                    if (sub->owner) {
                        p->sub = sub;
                    } else if (sub->next->owner) {
                        p->sub = sub->next;
                    } else {
                        /* We seem to have lost both our calls */
                        /* XXX - What do we do now? */
                        return -1;
                    }
                    if (p->sub->owner->bridge) {
                        ast_moh_stop(p->sub->owner->bridge);
                    }
                    p->sub->cxmode = MGCP_CX_SENDRECV;
                    transmit_modify_request(p->sub);
                }
            } else {
                ast_log(LOG_WARNING, "Callwaiting, call transfer or threeway calling not enabled on endpoint %s@%s\n", 
                        p->name, p->parent->name);
            }
            /* ast_moh_stop(sub->owner->bridge); */
		} else if (!strcasecmp(ev, "hu")) {
            p->hookstate = MGCP_ONHOOK;
            sub->cxmode = MGCP_CX_RECVONLY;
            ast_log(LOG_DEBUG, "MGCP %s@%s Went on hook\n", p->name, p->parent->name);
            if (sub->rtp) {
                transmit_modify_request(sub);
            }
            if (p->transfer && (sub->owner && sub->next->owner) && ((!sub->outgoing) || (!sub->next->outgoing))) {
                /* We're allowed to transfer, we have two avtive calls and */
                /* we made at least one of the calls.  Let's try and transfer */
                if ((res = attempt_transfer(p)) < 0) {
                    if (p->sub->next->owner) {
                        sub->next->alreadygone = 1;
                        ast_queue_hangup(sub->next->owner,1);
                    }
                } else if (res) {
                    ast_log(LOG_WARNING, "Transfer attempt failed\n");
                    return -1;
                }
            } else {
                /* Hangup the current call */
                /* If there is another active call, mgcp_hangup will ring the phone with the other call */
                if (sub->owner) {
                    sub->alreadygone = 1;
                    ast_queue_hangup(sub->owner, 1);
                } else {
                    ast_verbose(VERBOSE_PREFIX_3 "MGCP handle_request(%s@%s-%d) ast_channel already destroyed\n", 
                                p->name, p->parent->name, sub->id);
                }
            }
            if ((p->hookstate == MGCP_ONHOOK) && (!sub->rtp) && (!sub->next->rtp)) {
                if (has_voicemail(p)) {
                    if (option_verbose > 2) {
                        ast_verbose(VERBOSE_PREFIX_3 "MGCP handle_request(%s@%s) set vmwi(+)\n", p->name, p->parent->name);
                    }
                    transmit_notify_request(sub, "vmwi(+)");
                } else {
                    if (option_verbose > 2) {
                        ast_verbose(VERBOSE_PREFIX_3 "MGCP handle_request(%s@%s) set vmwi(-)\n", p->name, p->parent->name);
                    }
                    transmit_notify_request(sub, "vmwi(-)");
                }
            }
		} else if ((strlen(ev) == 1) && 
					(((ev[0] >= '0') && (ev[0] <= '9')) ||
					 ((ev[0] >= 'A') && (ev[0] <= 'D')) ||
					 (ev[0] == '*') || (ev[0] == '#'))) {
			f.frametype = AST_FRAME_DTMF;
			f.subclass = ev[0];
			f.src = "mgcp";
			if (sub->owner) {
                /* XXX MUST queue this frame to all subs in threeway call if threeway call is active */
				ast_queue_frame(sub->owner, &f, 1);
                if (sub->next->owner) {
                    ast_queue_frame(sub->next->owner, &f, 1);
                }
            }
            if (strstr(p->curtone, "wt") && (ev[0] == 'A')) {
                memset(p->curtone, 0, sizeof(p->curtone));
            }
		} else if (!strcasecmp(ev, "T")) {
			/* Digit timeout -- unimportant */
		} else {
			ast_log(LOG_NOTICE, "Received unknown event '%s' from %s@%s\n", ev, p->name, p->parent->name);
		}
	} else {
		ast_log(LOG_WARNING, "Unknown verb '%s' received from %s\n", req->verb, inet_ntoa(sin->sin_addr));
		transmit_response(sub, "510", req, "Unknown verb");
	}
	return 0;
}

static int mgcpsock_read(int *id, int fd, short events, void *ignore)
{
	struct mgcp_request req;
	struct sockaddr_in sin;
	struct mgcp_subchannel *sub;
	char *c;
	int res;
	int len;
	int result;
	int ident;
	len = sizeof(sin);
	memset(&req, 0, sizeof(req));
	res = recvfrom(mgcpsock, req.data, sizeof(req.data) - 1, 0, (struct sockaddr *)&sin, &len);
	if (res < 0) {
		if (errno != ECONNREFUSED)
			ast_log(LOG_WARNING, "Recv error: %s\n", strerror(errno));
		return 1;
	}
	req.data[res] = '\0';
	req.len = res;
	if (mgcpdebug) {
		ast_verbose("MGCP read: \n%s\nfrom %s:%d", req.data, inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
    }
	parse(&req);
	if (req.headers < 1) {
		/* Must have at least one header */
		return 1;
	}
	if (!req.identifier || !strlen(req.identifier)) {
		ast_log(LOG_NOTICE, "Message from %s missing identifier\n", inet_ntoa(sin.sin_addr));
		return 1;
	}

	if (sscanf(req.verb, "%d", &result) &&
		sscanf(req.identifier, "%d", &ident)) {
		/* Try to find who this message is for, if it's important */
		sub = find_subchannel(NULL, ident, &sin);
		if (sub) {
#if 0
			if ((c = get_header(&req, "X"))) {
                if (strlen(c)) {
                    if ((strcasecmp(sub->txident, c)) && (!strcasecmp(sub->next->txident, c))) {
                        ast_log(LOG_WARNING, "Response on sub%d message %d appears to have come from out other subchannel", sub->id, ident);
                    }
                }
            }
#endif
			handle_response(sub, result, ident);

			if ((c = get_header(&req, "I"))) {
				if (strlen(c)) {
                    if (strlen(sub->cxident)) {
                        if (strcasecmp(c, sub->cxident)) {
                            ast_log(LOG_WARNING, "Subchannel already has a cxident. sub->cxident: %s requested %s\n", sub->cxident, c);
                        }
                    }
					strncpy(sub->cxident, c, sizeof(sub->cxident) - 1);
					if (sub->tmpdest.sin_addr.s_addr) {
						transmit_modify_with_sdp(sub, NULL);
					}
				}
			}
            /* Try to determine the hookstate returned from an audit endpoint command */
			if ((c = get_header(&req, "ES"))) {
				if (strlen(c)) {
                    if (strstr(c, "hu")) {
                        sub->parent->hookstate = MGCP_ONHOOK;
                        ast_verbose(VERBOSE_PREFIX_3 "Setting hookstate of %s@%s to ONHOOK\n", sub->parent->name, sub->parent->parent->name);
                    } else if (strstr(c, "hd")) {
                        sub->parent->hookstate = MGCP_OFFHOOK;
                        ast_verbose(VERBOSE_PREFIX_3 "Setting hookstate of %s@%s to OFFHOOK\n", sub->parent->name, sub->parent->parent->name);
                    }
				}
			}

			if (req.lines) {
				if (!sub->rtp) 
					start_rtp(sub);
				if (sub->rtp)
					process_sdp(sub, &req);
			}

		}
	} else {
		if (!req.endpoint || !strlen(req.endpoint) || 
		    !req.version || !strlen(req.version) || 
			!req.verb || !strlen(req.verb)) {
			ast_log(LOG_NOTICE, "Message must have a verb, an idenitifier, version, and endpoint\n");
			return 1;
		}
		/* Process request, with iflock held */
		sub = find_subchannel(req.endpoint, 0, &sin);
		if (sub) {
            /* pass the request off to the currently mastering subchannel */
			handle_request(sub, &req, &sin);
		}
	}
	return 1;
}

static void *do_monitor(void *data)
{
	int res;
	struct mgcp_pkt *p;
    /* struct mgcp_gateway *g; */
    /* struct mgcp_endpoint *e; */
	/*time_t thispass = 0, lastpass = 0;*/

	sched = sched_context_create();
	if (!sched) {
		ast_log(LOG_WARNING, "Unable to create schedule context\n");
		return NULL;
	}
	io = io_context_create();
	if (!io) {
		ast_log(LOG_WARNING, "Unable to create I/O context\n");
		return NULL;
	}
	
	/* Add an I/O event to our UDP socket */
	if (mgcpsock > -1) 
		ast_io_add(io, mgcpsock, mgcpsock_read, AST_IO_IN, NULL);
	
	/* This thread monitors all the frame relay interfaces which are not yet in use
	   (and thus do not have a separate thread) indefinitely */
	/* From here on out, we die whenever asked */
	for(;;) {
		/* Check for interfaces needing to be killed */
		/* Don't let anybody kill us right away.  Nobody should lock the interface list
		   and wait for the monitor list, but the other way around is okay. */
		ast_mutex_lock(&monlock);
		/* Lock the network interface */
		ast_mutex_lock(&netlock);
		p = packets;
		while(p) {
			/* Handle any retransmissions */
			p = p->next;
		}

        /* XXX THIS IS COMPLETELY HOSED */
        /* The gateway goes into a state of panic */
        /* If the vmwi indicator is sent while it is reseting interfaces */
#if 0
        lastpass = thispass;
        thispass = time(NULL);
        g = gateways;
        while(g) {
            if (thispass != lastpass) {
                e = g->endpoints;
                while(e) {
                    if (e->type == TYPE_LINE) {
                        res = has_voicemail(e);
                        if ((e->msgstate != res) && (e->hookstate == MGCP_ONHOOK) && (!e->rtp)){
                            if (res) {
                                transmit_notify_request(e, "vmwi(+)");
                            } else {
                                transmit_notify_request(e, "vmwi(-)");
                            }
                            e->msgstate = res;
                            e->onhooktime = thispass;
                        }
                    }
                    e = e->next;
                }
            }
            g = g->next;
        }
#endif
        
		/* Okay, now that we know what to do, release the network lock */
		ast_mutex_unlock(&netlock);
		/* And from now on, we're okay to be killed, so release the monitor lock as well */
		ast_mutex_unlock(&monlock);
		pthread_testcancel();
		/* Wait for sched or io */
		res = ast_sched_wait(sched);
		res = ast_io_wait(io, res);
		ast_mutex_lock(&monlock);
		if (res >= 0) 
			ast_sched_runq(sched);
		ast_mutex_unlock(&monlock);
	}
	/* Never reached */
	return NULL;
	
}

static int restart_monitor(void)
{
	pthread_attr_t attr;
	pthread_attr_init(&attr);
        pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);	

	/* If we're supposed to be stopped -- stay stopped */
	if (monitor_thread == (pthread_t) -2)
		return 0;
	if (ast_mutex_lock(&monlock)) {
		ast_log(LOG_WARNING, "Unable to lock monitor\n");
		return -1;
	}
	if (monitor_thread == pthread_self()) {
		ast_mutex_unlock(&monlock);
		ast_log(LOG_WARNING, "Cannot kill myself\n");
		return -1;
	}
	if (monitor_thread) {
		/* Wake up the thread */
		pthread_kill(monitor_thread, SIGURG);
	} else {
		/* Start a new monitor */
		if (pthread_create(&monitor_thread, &attr, do_monitor, NULL) < 0) {
			ast_mutex_unlock(&monlock);
			ast_log(LOG_ERROR, "Unable to start monitor thread.\n");
			return -1;
		}
	}
	ast_mutex_unlock(&monlock);
	return 0;
}

static struct ast_channel *mgcp_request(char *type, int format, void *data)
{
	int oldformat;
	struct mgcp_subchannel *sub;
	struct ast_channel *tmpc = NULL;
	char tmp[256];
	char *dest = data;

	oldformat = format;
	format &= capability;
	if (!format) {
		ast_log(LOG_NOTICE, "Asked to get a channel of unsupported format '%d'\n", format);
		return NULL;
	}
	strncpy(tmp, dest, sizeof(tmp) - 1);
	if (!strlen(tmp)) {
		ast_log(LOG_NOTICE, "MGCP Channels require an endpoint\n");
		return NULL;
	}
	sub = find_subchannel(tmp, 0, NULL);
	if (!sub) {
		ast_log(LOG_WARNING, "Unable to find MGCP endpoint '%s'\n", tmp);
		return NULL;
	}
	
    if (option_verbose > 2) {
        ast_verbose(VERBOSE_PREFIX_3 "MGCP mgcp_request(%s)\n", tmp);
        ast_verbose(VERBOSE_PREFIX_3 "MGCP cw: %d, dnd: %d, so: %d, sno: %d\n", 
                    sub->parent->callwaiting, sub->parent->dnd, sub->owner ? 1 : 0, sub->next->owner ? 1: 0);
    }
	/* Must be busy */
	if (((sub->parent->callwaiting) && ((sub->owner) && (sub->next->owner))) ||
        ((!sub->parent->callwaiting) && (sub->owner)) ||
         (sub->parent->dnd && (!strlen(sub->parent->call_forward)))) {
         if (sub->parent->hookstate == MGCP_ONHOOK) {
             if (has_voicemail(sub->parent)) {
                 transmit_notify_request(sub,"vmwi(+)");
             } else {
                 transmit_notify_request(sub,"vmwi(-)");
             }
         }
		return NULL;
    }
	tmpc = mgcp_new(sub->owner ? sub->next : sub, AST_STATE_DOWN);
	if (!tmpc)
		ast_log(LOG_WARNING, "Unable to make channel for '%s'\n", tmp);
	restart_monitor();
	return tmpc;
}

static struct mgcp_gateway *build_gateway(char *cat, struct ast_variable *v)
{
	struct mgcp_gateway *gw;
	struct mgcp_endpoint *e;
    struct mgcp_subchannel *sub;
    /*char txident[80];*/
    int i=0, y=0;
	canreinvite = CANREINVITE;
	gw = malloc(sizeof(struct mgcp_gateway));
	if (gw) {
		memset(gw, 0, sizeof(struct mgcp_gateway));
		gw->expire = -1;
		strncpy(gw->name, cat, sizeof(gw->name) - 1);
		while(v) {
			if (!strcasecmp(v->name, "host")) {
				if (!strcasecmp(v->value, "dynamic")) {
					/* They'll register with us */
					gw->dynamic = 1;
					memset(&gw->addr.sin_addr, 0, 4);
					if (gw->addr.sin_port) {
						/* If we've already got a port, make it the default rather than absolute */
						gw->defaddr.sin_port = gw->addr.sin_port;
						gw->addr.sin_port = 0;
					}
				} else {
					/* Non-dynamic.  Make sure we become that way if we're not */
					if (gw->expire > -1)
						ast_sched_del(sched, gw->expire);
					gw->expire = -1;
					gw->dynamic = 0;
					if (ast_get_ip(&gw->addr, v->value)) {
						free(gw);
						return NULL;
					}
				}
			} else if (!strcasecmp(v->name, "defaultip")) {
				if (ast_get_ip(&gw->defaddr, v->value)) {
					free(gw);
					return NULL;
				}
			} else if (!strcasecmp(v->name, "permit") ||
					   !strcasecmp(v->name, "deny")) {
				gw->ha = ast_append_ha(v->name, v->value, gw->ha);
			} else if (!strcasecmp(v->name, "port")) {
				gw->addr.sin_port = htons(atoi(v->value));
			} else if (!strcasecmp(v->name, "context")) {
				strncpy(context, v->value, sizeof(context) - 1);
			} else if (!strcasecmp(v->name, "inbanddtmf")) {
				inbanddtmf = atoi(v->value);
			} else if (!strcasecmp(v->name, "nat")) {
				nat = ast_true(v->value);
			} else if (!strcasecmp(v->name, "callerid")) {
				if (!strcasecmp(v->value, "asreceived"))
					strcpy(callerid, "");
				else
					strncpy(callerid, v->value, sizeof(callerid) - 1);
			} else if (!strcasecmp(v->name, "language")) {
				strncpy(language, v->value, sizeof(language)-1);
            } else if (!strcasecmp(v->name, "accountcode")) {
                strncpy(accountcode, v->value, sizeof(accountcode)-1);
            } else if (!strcasecmp(v->name, "amaflags")) {
                y = ast_cdr_amaflags2int(v->value);
                if (y < 0) {
                    ast_log(LOG_WARNING, "Invalid AMA flags: %s at line %d\n", v->value, v->lineno);
                } else {
                    amaflags = y;
                }
            } else if (!strcasecmp(v->name, "musiconhold")) {
                strncpy(musicclass, v->value, sizeof(musicclass)-1);
            } else if (!strcasecmp(v->name, "callgroup")) {
                cur_callergroup = ast_get_group(v->value);
            } else if (!strcasecmp(v->name, "pickupgroup")) {
                cur_pickupgroup = ast_get_group(v->value);
            } else if (!strcasecmp(v->name, "immediate")) {
                immediate = ast_true(v->value);
            } else if (!strcasecmp(v->name, "cancallforward")) {
                cancallforward = ast_true(v->value);
            } else if (!strcasecmp(v->name, "canreinvite")) {
                canreinvite = ast_true(v->value);
            } else if (!strcasecmp(v->name, "mailbox")) {
                strncpy(mailbox, v->value, sizeof(mailbox) -1);
            } else if (!strcasecmp(v->name, "adsi")) {
                adsi = ast_true(v->value);
            } else if (!strcasecmp(v->name, "callreturn")) {
                callreturn = ast_true(v->value);
            } else if (!strcasecmp(v->name, "immediate")) {
                immediate = ast_true(v->value);
            } else if (!strcasecmp(v->name, "callwaiting")) {
                callwaiting = ast_true(v->value);
            } else if (!strcasecmp(v->name, "transfer")) {
                transfer = ast_true(v->value);
            } else if (!strcasecmp(v->name, "threewaycalling")) {
                threewaycalling = ast_true(v->value);

			} else if (!strcasecmp(v->name, "trunk") ||
			           !strcasecmp(v->name, "line")) {
				e = malloc(sizeof(struct mgcp_endpoint));
				if (e) {
					memset(e, 0, sizeof(struct mgcp_endpoint));
					strncpy(e->name, v->value, sizeof(e->name) - 1);
					/* XXX Should we really check for uniqueness?? XXX */
					strncpy(e->context, context, sizeof(e->context) - 1);
					strncpy(e->callerid, callerid, sizeof(e->callerid) - 1);
					strncpy(e->language, language, sizeof(e->language) - 1);
                    strncpy(e->musicclass, musicclass, sizeof(e->musicclass)-1);
                    strncpy(e->mailbox, mailbox, sizeof(e->mailbox)-1);
                    if (strlen(mailbox)) {
                        ast_verbose(VERBOSE_PREFIX_3 "Setting mailbox '%s' on %s@%s\n", mailbox, gw->name, e->name);
                    }
                    e->msgstate = -1;
					e->capability = capability;
					e->parent = gw;
					e->dtmfinband = inbanddtmf;
					e->adsi = adsi;
					if (!strcasecmp(v->name, "trunk"))
						e->type = TYPE_TRUNK;
					else
						e->type = TYPE_LINE;

                    e->immediate = immediate;
                    e->callgroup=cur_callergroup;
                    e->pickupgroup=cur_pickupgroup;
                    e->callreturn = callreturn;
                    e->cancallforward = cancallforward;
                    e->canreinvite = canreinvite;
                    e->callwaiting = callwaiting;
                    e->transfer = transfer;
                    e->threewaycalling = threewaycalling;
                    e->onhooktime = time(NULL);
                    /* ASSUME we're onhook */
                    e->hookstate = MGCP_ONHOOK;
                    /*snprintf(txident, sizeof(txident), "%08x", rand());*/

                    for (i = 0; i < MAX_SUBS; i++) {
                        sub = malloc(sizeof(struct mgcp_subchannel));
                        if (sub) {
                            ast_verbose(VERBOSE_PREFIX_3 "Allocating subchannel '%d' on %s@%s\n", i, e->name, gw->name);
                            memset(sub, 0, sizeof(struct mgcp_subchannel));
                            sub->parent = e;
                            sub->id = i;
                            snprintf(sub->txident, sizeof(sub->txident), "%08x", rand());
                            /*strcpy(sub->txident, txident);*/
                            sub->cxmode = MGCP_CX_INACTIVE;
                            sub->nat = nat;
                            sub->next = e->sub;
                            e->sub = sub;
                        } else {
                            /* XXX Should find a way to clean up our memory */
                            ast_log(LOG_WARNING, "Out of memory allocating subchannel");
                            return NULL;
                        }
                    }
                    /* Make out subs a circular linked list so we can always sping through the whole bunch */
                    sub = e->sub;
                    /* find the end of the list */
                    while(sub->next){
                        sub = sub->next;
                    }
                    /* set the last sub->next to the first sub */
                    sub->next = e->sub;


					e->next = gw->endpoints;
					gw->endpoints = e;
				}
			} else
				ast_log(LOG_WARNING, "Don't know keyword '%s' at line %d\n", v->name, v->lineno);
			v = v->next;
		}
		
	}
	if (!ntohl(gw->addr.sin_addr.s_addr) && !gw->dynamic) {
		ast_log(LOG_WARNING, "Gateway '%s' lacks IP address and isn't dynamic\n", gw->name);
		free(gw);
		return NULL;
	}
	if (gw->defaddr.sin_addr.s_addr && !ntohs(gw->defaddr.sin_port)) 
		gw->defaddr.sin_port = htons(DEFAULT_MGCP_PORT);
	if (gw->addr.sin_addr.s_addr && !ntohs(gw->addr.sin_port))
		gw->addr.sin_port = htons(DEFAULT_MGCP_PORT);
	if (gw->addr.sin_addr.s_addr)
		if (ast_ouraddrfor(&gw->addr.sin_addr, &gw->ourip))
			memcpy(&gw->ourip, &__ourip, sizeof(gw->ourip));
	return gw;
}

static struct ast_rtp *mgcp_get_rtp_peer(struct ast_channel *chan)
{
	struct mgcp_subchannel *sub;
	sub = chan->pvt->pvt;
	if (sub && sub->rtp && sub->parent->canreinvite)
		return sub->rtp;
	return NULL;
}

static int mgcp_set_rtp_peer(struct ast_channel *chan, struct ast_rtp *rtp, struct ast_rtp *vrtp)
{
	/* XXX Is there such thing as video support with MGCP? XXX */
	struct mgcp_subchannel *sub;
	sub = chan->pvt->pvt;
	if (sub) {
		transmit_modify_with_sdp(sub, rtp);
		return 0;
	}
	return -1;
}

static struct ast_rtp_protocol mgcp_rtp = {
	get_rtp_info: mgcp_get_rtp_peer,
	set_rtp_peer: mgcp_set_rtp_peer,
};

static int mgcp_do_debug(int fd, int argc, char *argv[])
{
	if (argc != 2)
		return RESULT_SHOWUSAGE;
	mgcpdebug = 1;
	ast_cli(fd, "MGCP Debugging Enabled\n");
	return RESULT_SUCCESS;
}

static int mgcp_no_debug(int fd, int argc, char *argv[])
{
	if (argc != 3)
		return RESULT_SHOWUSAGE;
	mgcpdebug = 0;
	ast_cli(fd, "MGCP Debugging Disabled\n");
	return RESULT_SUCCESS;
}

static char debug_usage[] = 
"Usage: mgcp debug\n"
"       Enables dumping of MGCP packets for debugging purposes\n";

static char no_debug_usage[] = 
"Usage: mgcp no debug\n"
"       Disables dumping of MGCP packets for debugging purposes\n";

static struct ast_cli_entry  cli_debug =
	{ { "mgcp", "debug", NULL }, mgcp_do_debug, "Enable MGCP debugging", debug_usage };
static struct ast_cli_entry  cli_no_debug =
	{ { "mgcp", "no", "debug", NULL }, mgcp_no_debug, "Disable MGCP debugging", no_debug_usage };


int load_module()
{
	struct ast_config *cfg;
	struct ast_variable *v;
	struct mgcp_gateway *g;
	struct mgcp_endpoint *e;
	char *cat;
	struct hostent *hp;
	int format;
	
	if (gethostname(ourhost, sizeof(ourhost))) {
		ast_log(LOG_WARNING, "Unable to get hostname, MGCP disabled\n");
		return 0;
	}
	cfg = ast_load(config);

	/* We *must* have a config file otherwise stop immediately */
	if (!cfg) {
		ast_log(LOG_NOTICE, "Unable to load config %s, MGCP disabled\n", config);
		return 0;
	}
	memset(&bindaddr, 0, sizeof(bindaddr));
	v = ast_variable_browse(cfg, "general");
	while(v) {
		/* Create the interface list */
		if (!strcasecmp(v->name, "bindaddr")) {
			if (!(hp = gethostbyname(v->value))) {
				ast_log(LOG_WARNING, "Invalid address: %s\n", v->value);
			} else {
				memcpy(&bindaddr.sin_addr, hp->h_addr, sizeof(bindaddr.sin_addr));
			}
		} else if (!strcasecmp(v->name, "allow")) {
			format = ast_getformatbyname(v->value);
			if (format < 1) 
				ast_log(LOG_WARNING, "Cannot allow unknown format '%s'\n", v->value);
			else
				capability |= format;
		} else if (!strcasecmp(v->name, "disallow")) {
			format = ast_getformatbyname(v->value);
			if (format < 1) 
				ast_log(LOG_WARNING, "Cannot disallow unknown format '%s'\n", v->value);
			else
				capability &= ~format;
		} else if (!strcasecmp(v->name, "tos")) {
			if (sscanf(v->value, "%i", &format) == 1)
				tos = format & 0xff;
			else if (!strcasecmp(v->value, "lowdelay"))
				tos = IPTOS_LOWDELAY;
			else if (!strcasecmp(v->value, "throughput"))
				tos = IPTOS_THROUGHPUT;
			else if (!strcasecmp(v->value, "reliability"))
				tos = IPTOS_RELIABILITY;
			else if (!strcasecmp(v->value, "mincost"))
				tos = IPTOS_MINCOST;
			else if (!strcasecmp(v->value, "none"))
				tos = 0;
			else
				ast_log(LOG_WARNING, "Invalid tos value at line %d, should be 'lowdelay', 'throughput', 'reliability', 'mincost', or 'none'\n", v->lineno);
		} else if (!strcasecmp(v->name, "port")) {
			if (sscanf(v->value, "%i", &ourport) == 1) {
				bindaddr.sin_port = htons(ourport);
			} else {
				ast_log(LOG_WARNING, "Invalid port number '%s' at line %d of %s\n", v->value, v->lineno, config);
			}
		}
		v = v->next;
	}
	
	cat = ast_category_browse(cfg, NULL);
	while(cat) {
		if (strcasecmp(cat, "general")) {
			g = build_gateway(cat, ast_variable_browse(cfg, cat));
			if (g) {
				if (option_verbose > 2) {
					ast_verbose(VERBOSE_PREFIX_3 "Added gateway '%s'\n", g->name);
                }
				ast_mutex_lock(&gatelock);
				g->next = gateways;
				gateways = g;
				ast_mutex_unlock(&gatelock);
			}
		}
		cat = ast_category_browse(cfg, cat);
	}
	
	if (ntohl(bindaddr.sin_addr.s_addr)) {
		memcpy(&__ourip, &bindaddr.sin_addr, sizeof(__ourip));
	} else {
		hp = gethostbyname(ourhost);
		if (!hp) {
			ast_log(LOG_WARNING, "Unable to get our IP address, MGCP disabled\n");
			return 0;
		}
		memcpy(&__ourip, hp->h_addr, sizeof(__ourip));
	}
	if (!ntohs(bindaddr.sin_port))
		bindaddr.sin_port = ntohs(DEFAULT_MGCP_PORT);
	bindaddr.sin_family = AF_INET;
	ast_mutex_lock(&netlock);
	if (mgcpsock > -1)
		close(mgcpsock);
	mgcpsock = socket(AF_INET, SOCK_DGRAM, 0);
	if (mgcpsock < 0) {
		ast_log(LOG_WARNING, "Unable to create MGCP socket: %s\n", strerror(errno));
	} else {
		if (bind(mgcpsock, (struct sockaddr *)&bindaddr, sizeof(bindaddr)) < 0) {
			ast_log(LOG_WARNING, "Failed to bind to %s:%d: %s\n",
					inet_ntoa(bindaddr.sin_addr), ntohs(bindaddr.sin_port),
						strerror(errno));
			close(mgcpsock);
			mgcpsock = -1;
		} else {
			if (option_verbose > 1) {
				ast_verbose(VERBOSE_PREFIX_2 "MGCP Listening on %s:%d\n", 
					inet_ntoa(bindaddr.sin_addr), ntohs(bindaddr.sin_port));
				ast_verbose(VERBOSE_PREFIX_2 "Using TOS bits %d\n", tos);
			}
			if (setsockopt(mgcpsock, IPPROTO_IP, IP_TOS, &tos, sizeof(tos))) 
				ast_log(LOG_WARNING, "Unable to set TOS to %d\n", tos);
        }
	}
	ast_mutex_unlock(&netlock);
	ast_destroy(cfg);

	/* Make sure we can register our mgcp channel type */
	if (ast_channel_register(type, tdesc, capability, mgcp_request)) {
		ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
		ast_destroy(cfg);
		return -1;
	}
	mgcp_rtp.type = type;
	ast_rtp_proto_register(&mgcp_rtp);
	ast_cli_register(&cli_show_endpoints);
	ast_cli_register(&cli_audit_endpoint);
	ast_cli_register(&cli_debug);
	ast_cli_register(&cli_no_debug);
	/* And start the monitor for the first time */
	restart_monitor();

    g = gateways;
    while (g) {
		e = g->endpoints;
        while (e) {
            transmit_audit_endpoint(e);
			ast_verbose(VERBOSE_PREFIX_3 "MGCP Auditing endpoint %s@%s for hookstate\n", e->name, g->name);
            e = e->next;
        }
        g = g->next;
    }

	return 0;
}

int unload_module()
{
#if 0
	struct mgcp_endpoint *p, *pl;
	/* First, take us out of the channel loop */
	ast_channel_unregister(type);
	if (!ast_mutex_lock(&gatelock)) {
		/* Hangup all interfaces if they have an owner */
		p = iflist;
		while(p) {
			if (p->owner)
				ast_softhangup(p->owner, AST_SOFTHANGUP_APPUNLOAD);
			p = p->next;
		}
		iflist = NULL;
		ast_mutex_unlock(&iflock);
	} else {
		ast_log(LOG_WARNING, "Unable to lock the monitor\n");
		return -1;
	}
	if (!ast_mutex_lock(&monlock)) {
		if (monitor_thread && (monitor_thread != -2)) {
			pthread_cancel(monitor_thread);
			pthread_kill(monitor_thread, SIGURG);
			pthread_join(monitor_thread, NULL);
		}
		monitor_thread = -2;
		ast_mutex_unlock(&monlock);
	} else {
		ast_log(LOG_WARNING, "Unable to lock the monitor\n");
		return -1;
	}

	if (!ast_mutex_lock(&iflock)) {
		/* Destroy all the interfaces and free their memory */
		p = iflist;
		while(p) {
			pl = p;
			p = p->next;
			/* Free associated memory */
			free(pl);
		}
		iflist = NULL;
		ast_mutex_unlock(&iflock);
	} else {
		ast_log(LOG_WARNING, "Unable to lock the monitor\n");
		return -1;
	}
#endif		
	return -1;
}

int usecount()
{
	int res;
	ast_mutex_lock(&usecnt_lock);
	res = usecnt;
	ast_mutex_unlock(&usecnt_lock);
	return res;
}

char *key()
{
	return ASTERISK_GPL_KEY;
}

char *description()
{
	return desc;
}

